<?php

namespace App\Services;

use App\Models\Project;
use App\Models\ProjectMember;
use Illuminate\Support\Facades\DB;

class ProjectService
{
    public function create(int $tenantId, array $data, int $userId): Project
    {
        return DB::transaction(function () use ($tenantId, $data, $userId) {
            $project = Project::create([
                'tenant_id' => $tenantId,
                'workspace_id' => $data['workspace_id'] ?? null,
                'contact_id' => $data['contact_id'] ?? null,
                'name' => $data['name'],
                'description' => $data['description'] ?? null,
                'color' => $data['color'] ?? '#2196F3',
                'icon' => $data['icon'] ?? null,
                'start_date' => $data['start_date'] ?? null,
                'end_date' => $data['end_date'] ?? null,
                'budget_hours' => $data['budget_hours'] ?? null,
                'budget_amount' => $data['budget_amount'] ?? null,
                'hourly_rate' => $data['hourly_rate'] ?? null,
                'is_billable' => $data['is_billable'] ?? true,
                'is_template' => $data['is_template'] ?? false,
                'visibility' => $data['visibility'] ?? 'team',
                'created_by' => $userId,
            ]);

            // Add creator as admin
            $project->members()->create([
                'tenant_id' => $tenantId,
                'user_id' => $userId,
                'role' => ProjectMember::ROLE_ADMIN,
            ]);

            // Add additional members
            if (!empty($data['members'])) {
                foreach ($data['members'] as $member) {
                    if ($member['user_id'] !== $userId) {
                        $project->members()->create([
                            'tenant_id' => $tenantId,
                            'user_id' => $member['user_id'],
                            'role' => $member['role'] ?? ProjectMember::ROLE_MEMBER,
                            'hourly_rate' => $member['hourly_rate'] ?? null,
                        ]);
                    }
                }
            }

            // Create default task statuses
            $this->createDefaultStatuses($project);

            return $project;
        });
    }

    protected function createDefaultStatuses(Project $project): void
    {
        $statuses = [
            ['name' => 'To Do', 'color' => '#9E9E9E', 'position' => 1, 'is_default' => true],
            ['name' => 'In Progress', 'color' => '#2196F3', 'position' => 2],
            ['name' => 'In Review', 'color' => '#FF9800', 'position' => 3],
            ['name' => 'Done', 'color' => '#4CAF50', 'position' => 4, 'is_completed' => true],
        ];

        foreach ($statuses as $status) {
            $project->statuses()->create(array_merge($status, [
                'tenant_id' => $project->tenant_id,
            ]));
        }
    }

    public function update(Project $project, array $data): Project
    {
        $project->update(array_filter([
            'workspace_id' => $data['workspace_id'] ?? null,
            'contact_id' => $data['contact_id'] ?? null,
            'name' => $data['name'] ?? null,
            'description' => $data['description'] ?? null,
            'color' => $data['color'] ?? null,
            'icon' => $data['icon'] ?? null,
            'start_date' => $data['start_date'] ?? null,
            'end_date' => $data['end_date'] ?? null,
            'budget_hours' => $data['budget_hours'] ?? null,
            'budget_amount' => $data['budget_amount'] ?? null,
            'hourly_rate' => $data['hourly_rate'] ?? null,
            'is_billable' => $data['is_billable'] ?? null,
            'visibility' => $data['visibility'] ?? null,
        ], fn($v) => !is_null($v)));

        return $project->fresh();
    }

    public function addMember(Project $project, int $userId, string $role = 'member', ?float $hourlyRate = null): ProjectMember
    {
        return $project->members()->updateOrCreate(
            ['user_id' => $userId],
            [
                'tenant_id' => $project->tenant_id,
                'role' => $role,
                'hourly_rate' => $hourlyRate,
            ]
        );
    }

    public function removeMember(Project $project, int $userId): void
    {
        $project->members()->where('user_id', $userId)->delete();
    }

    public function archive(Project $project): void
    {
        $project->update(['status' => Project::STATUS_ARCHIVED]);
    }

    public function restore(Project $project): void
    {
        $project->update(['status' => Project::STATUS_ACTIVE]);
    }

    public function duplicate(Project $project, int $userId, bool $includeTasks = false): Project
    {
        return DB::transaction(function () use ($project, $userId, $includeTasks) {
            $newProject = $project->replicate(['status']);
            $newProject->name = $project->name . ' (Copy)';
            $newProject->status = Project::STATUS_ACTIVE;
            $newProject->created_by = $userId;
            $newProject->save();

            // Copy members
            foreach ($project->members as $member) {
                $newProject->members()->create([
                    'tenant_id' => $project->tenant_id,
                    'user_id' => $member->user_id,
                    'role' => $member->role,
                    'hourly_rate' => $member->hourly_rate,
                ]);
            }

            // Copy statuses
            foreach ($project->statuses as $status) {
                $newProject->statuses()->create([
                    'tenant_id' => $project->tenant_id,
                    'name' => $status->name,
                    'color' => $status->color,
                    'position' => $status->position,
                    'is_default' => $status->is_default,
                    'is_completed' => $status->is_completed,
                ]);
            }

            if ($includeTasks) {
                // Copy tasks (without assignments)
                foreach ($project->tasks()->whereNull('parent_id')->get() as $task) {
                    $this->duplicateTask($task, $newProject, $userId);
                }
            }

            return $newProject;
        });
    }

    protected function duplicateTask($task, Project $newProject, int $userId): void
    {
        $newTask = $task->replicate(['completed_at', 'completed_by', 'actual_hours']);
        $newTask->project_id = $newProject->id;
        $newTask->created_by = $userId;
        $newTask->save();

        foreach ($task->subtasks as $subtask) {
            $this->duplicateTask($subtask, $newProject, $userId);
        }
    }
}
