<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\Payment;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Checkout\Session;

class StripeService
{
    public function __construct()
    {
        Stripe::setApiKey(config('services.stripe.secret'));
    }

    public function createPaymentIntent(Invoice $invoice): PaymentIntent
    {
        $amountDue = $invoice->getAmountDue();
        
        // Convert to smallest currency unit
        $amount = $invoice->currency === 'BHD' 
            ? (int) ($amountDue * 1000) // BHD has 3 decimal places
            : (int) ($amountDue * 100);

        return PaymentIntent::create([
            'amount' => $amount,
            'currency' => strtolower($invoice->currency),
            'metadata' => [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'tenant_id' => $invoice->tenant_id,
            ],
        ]);
    }

    public function createCheckoutSession(Invoice $invoice, string $successUrl, string $cancelUrl): Session
    {
        $amountDue = $invoice->getAmountDue();
        $amount = $invoice->currency === 'BHD' 
            ? (int) ($amountDue * 1000)
            : (int) ($amountDue * 100);

        return Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                    'currency' => strtolower($invoice->currency),
                    'unit_amount' => $amount,
                    'product_data' => [
                        'name' => "Invoice #{$invoice->invoice_number}",
                        'description' => "Payment for invoice #{$invoice->invoice_number}",
                    ],
                ],
                'quantity' => 1,
            ]],
            'mode' => 'payment',
            'success_url' => $successUrl,
            'cancel_url' => $cancelUrl,
            'metadata' => [
                'invoice_id' => $invoice->id,
                'tenant_id' => $invoice->tenant_id,
            ],
        ]);
    }

    public function handleWebhook(array $payload): void
    {
        $event = $payload['type'];
        $data = $payload['data']['object'];

        match ($event) {
            'payment_intent.succeeded' => $this->handlePaymentSuccess($data),
            'checkout.session.completed' => $this->handleCheckoutComplete($data),
            default => null,
        };
    }

    protected function handlePaymentSuccess(array $data): void
    {
        $invoiceId = $data['metadata']['invoice_id'] ?? null;
        
        if (!$invoiceId) {
            return;
        }

        $invoice = Invoice::find($invoiceId);
        
        if (!$invoice) {
            return;
        }

        $amount = $invoice->currency === 'BHD'
            ? $data['amount'] / 1000
            : $data['amount'] / 100;

        Payment::create([
            'tenant_id' => $invoice->tenant_id,
            'invoice_id' => $invoice->id,
            'contact_id' => $invoice->contact_id,
            'payment_date' => now(),
            'amount' => $amount,
            'currency' => $invoice->currency,
            'payment_method' => 'stripe',
            'gateway_transaction_id' => $data['id'],
            'status' => 'completed',
        ]);

        $invoice->updatePaymentStatus();
    }

    protected function handleCheckoutComplete(array $data): void
    {
        if ($data['payment_status'] === 'paid') {
            $this->handlePaymentSuccess([
                'amount' => $data['amount_total'],
                'id' => $data['payment_intent'],
                'metadata' => $data['metadata'],
            ]);
        }
    }

    public function refund(Payment $payment, ?float $amount = null): void
    {
        $refundAmount = $amount ?? $payment->amount;
        $stripeAmount = $payment->currency === 'BHD'
            ? (int) ($refundAmount * 1000)
            : (int) ($refundAmount * 100);

        \Stripe\Refund::create([
            'payment_intent' => $payment->gateway_transaction_id,
            'amount' => $stripeAmount,
        ]);

        $payment->update([
            'refunded_amount' => ($payment->refunded_amount ?? 0) + $refundAmount,
            'status' => $refundAmount >= $payment->amount ? 'refunded' : 'partially_refunded',
        ]);

        $payment->invoice?->updatePaymentStatus();
    }
}
