# 🤖 BOT 2: CRM MODULE

## YOUR MISSION
Build complete CRM: Contacts, Companies, Pipelines, Opportunities, Activities, Tags.
**Est. Files:** 50 | **Est. Lines:** 12,000

---

## 📁 CREATE THESE FILES

### MIGRATIONS (8 files)
```
database/migrations/
├── 2024_01_02_000001_create_contacts_table.php
├── 2024_01_02_000002_create_tags_table.php
├── 2024_01_02_000003_create_contact_tag_table.php
├── 2024_01_02_000004_create_pipelines_table.php
├── 2024_01_02_000005_create_pipeline_stages_table.php
├── 2024_01_02_000006_create_opportunities_table.php
├── 2024_01_02_000007_create_activities_table.php
├── 2024_01_02_000008_create_products_table.php
```

### MODELS (10 files)
```
app/Models/
├── Contact.php
├── Tag.php
├── Pipeline.php
├── PipelineStage.php
├── Opportunity.php
├── OpportunityProduct.php
├── Activity.php
├── Product.php
├── SmartList.php
├── CustomField.php
```

### CONTROLLERS (8 files)
```
app/Http/Controllers/Api/
├── ContactController.php
├── TagController.php
├── PipelineController.php
├── OpportunityController.php
├── ActivityController.php
├── ProductController.php
├── SmartListController.php
├── CrmDashboardController.php
```

### SERVICES (6 files)
```
app/Services/
├── ContactService.php
├── PipelineService.php
├── OpportunityService.php
├── ActivityService.php
├── ContactImportService.php
├── LeadScoringService.php
```

### REQUESTS (14 files)
```
app/Http/Requests/Contact/
├── StoreContactRequest.php
├── UpdateContactRequest.php
├── ImportContactsRequest.php
├── MergeContactsRequest.php

app/Http/Requests/Pipeline/
├── StorePipelineRequest.php
├── UpdatePipelineRequest.php

app/Http/Requests/Opportunity/
├── StoreOpportunityRequest.php
├── UpdateOpportunityRequest.php
├── MoveOpportunityRequest.php

app/Http/Requests/Activity/
├── StoreActivityRequest.php
├── UpdateActivityRequest.php

app/Http/Requests/Tag/
├── StoreTagRequest.php

app/Http/Requests/Product/
├── StoreProductRequest.php
```

### ROUTES (1 file)
```
routes/api/crm.php
```

### JOBS (4 files)
```
app/Jobs/
├── ImportContacts.php
├── ExportContacts.php
├── RecalculateLeadScores.php
├── UpdateOpportunityRotting.php
```

### EVENTS (5 files)
```
app/Events/
├── ContactCreated.php
├── ContactUpdated.php
├── OpportunityStageChanged.php
├── OpportunityWon.php
├── OpportunityLost.php
```

### SEEDERS (3 files)
```
database/seeders/
├── ContactSeeder.php
├── PipelineSeeder.php
├── ProductSeeder.php
```

### TESTS (6 files)
```
tests/Feature/
├── ContactTest.php
├── PipelineTest.php
├── OpportunityTest.php
├── ActivityTest.php
├── ContactImportTest.php
├── SmartListTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### Migration: create_contacts_table.php
```php
Schema::create('contacts', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('tenant_id');
    $table->uuid('owner_id')->nullable(); // Assigned user
    $table->uuid('company_id')->nullable(); // Parent company
    
    // Basic Info
    $table->string('first_name');
    $table->string('last_name')->nullable();
    $table->string('email')->nullable();
    $table->string('phone')->nullable();
    $table->string('mobile')->nullable();
    $table->string('whatsapp')->nullable();
    
    // Company Info
    $table->string('company_name')->nullable();
    $table->string('job_title')->nullable();
    $table->string('website')->nullable();
    
    // Address
    $table->text('address')->nullable();
    $table->string('city')->nullable();
    $table->string('state')->nullable();
    $table->string('postal_code')->nullable();
    $table->string('country')->nullable();
    
    // Status
    $table->enum('type', ['lead', 'customer', 'vendor', 'partner'])->default('lead');
    $table->enum('status', ['active', 'inactive', 'archived'])->default('active');
    $table->string('lead_status')->nullable(); // new, contacted, qualified, etc.
    $table->string('lead_source')->nullable();
    $table->integer('lead_score')->default(0);
    
    // Bahrain-specific
    $table->string('cr_number')->nullable(); // Commercial Registration
    $table->string('nationality')->nullable();
    $table->enum('ownership_type', ['100_foreign', '51_bahraini', 'gcc', 'mixed'])->nullable();
    $table->string('business_sector')->nullable();
    
    // Financial
    $table->decimal('lifetime_value', 15, 3)->default(0);
    $table->string('currency', 3)->default('BHD');
    
    // Dates
    $table->date('date_of_birth')->nullable();
    $table->timestamp('last_contacted_at')->nullable();
    $table->timestamp('converted_at')->nullable();
    
    // Custom fields
    $table->json('custom_fields')->nullable();
    
    $table->timestamps();
    $table->softDeletes();
    
    $table->foreign('tenant_id')->references('id')->on('tenants')->cascadeOnDelete();
    $table->foreign('owner_id')->references('id')->on('users')->nullOnDelete();
    
    $table->index(['tenant_id', 'email']);
    $table->index(['tenant_id', 'type', 'status']);
    $table->fullText(['first_name', 'last_name', 'email', 'company_name']);
});
```

### Migration: create_opportunities_table.php
```php
Schema::create('opportunities', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('tenant_id');
    $table->uuid('contact_id');
    $table->uuid('pipeline_id');
    $table->uuid('stage_id');
    $table->uuid('owner_id')->nullable();
    
    $table->string('name');
    $table->decimal('value', 15, 3)->default(0);
    $table->string('currency', 3)->default('BHD');
    $table->integer('probability')->default(0); // 0-100
    $table->decimal('expected_value', 15, 3)->default(0);
    
    $table->enum('status', ['open', 'won', 'lost'])->default('open');
    $table->string('lost_reason')->nullable();
    $table->date('expected_close_date')->nullable();
    $table->timestamp('won_at')->nullable();
    $table->timestamp('lost_at')->nullable();
    
    // Bahrain-specific packages
    $table->enum('package', ['standard', 'gold', 'premium'])->nullable();
    $table->string('legal_form')->nullable(); // WLL, SPC, Branch
    $table->string('business_activity')->nullable();
    
    $table->text('notes')->nullable();
    $table->integer('position')->default(0);
    $table->integer('days_in_stage')->default(0);
    $table->json('custom_fields')->nullable();
    
    $table->timestamps();
    $table->softDeletes();
    
    $table->foreign('tenant_id')->references('id')->on('tenants')->cascadeOnDelete();
    $table->foreign('contact_id')->references('id')->on('contacts')->cascadeOnDelete();
    $table->foreign('pipeline_id')->references('id')->on('pipelines')->cascadeOnDelete();
    $table->foreign('stage_id')->references('id')->on('pipeline_stages')->cascadeOnDelete();
});
```

### Model: Contact.php
```php
<?php
namespace App\Models;

use App\Traits\BelongsToTenant;
use Illuminate\Database\Eloquent\{Model, SoftDeletes};
use Illuminate\Database\Eloquent\Concerns\HasUuids;

class Contact extends Model
{
    use HasUuids, SoftDeletes, BelongsToTenant;
    
    protected $guarded = ['id'];
    
    protected $casts = [
        'custom_fields' => 'array',
        'lifetime_value' => 'decimal:3',
        'date_of_birth' => 'date',
        'last_contacted_at' => 'datetime',
        'converted_at' => 'datetime',
    ];
    
    // Relationships
    public function owner() { return $this->belongsTo(User::class, 'owner_id'); }
    public function company() { return $this->belongsTo(Contact::class, 'company_id'); }
    public function employees() { return $this->hasMany(Contact::class, 'company_id'); }
    public function tags() { return $this->belongsToMany(Tag::class, 'contact_tag'); }
    public function opportunities() { return $this->hasMany(Opportunity::class); }
    public function activities() { return $this->hasMany(Activity::class); }
    public function invoices() { return $this->hasMany(\App\Models\Invoice::class); }
    
    // Accessors
    public function getFullNameAttribute(): string {
        return trim("{$this->first_name} {$this->last_name}");
    }
    
    public function getDisplayNameAttribute(): string {
        return $this->company_name ?: $this->full_name;
    }
    
    // Scopes
    public function scopeLeads($query) { return $query->where('type', 'lead'); }
    public function scopeCustomers($query) { return $query->where('type', 'customer'); }
    public function scopeActive($query) { return $query->where('status', 'active'); }
    
    public function scopeSearch($query, string $search) {
        return $query->whereFullText(['first_name', 'last_name', 'email', 'company_name'], $search);
    }
    
    // Methods
    public function convertToCustomer(): void {
        $this->update([
            'type' => 'customer',
            'converted_at' => now(),
        ]);
    }
    
    public function updateLifetimeValue(): void {
        $this->update([
            'lifetime_value' => $this->invoices()->where('status', 'paid')->sum('total'),
        ]);
    }
    
    public function logActivity(string $type, array $data = []): Activity {
        return $this->activities()->create([
            'tenant_id' => $this->tenant_id,
            'user_id' => auth()->id(),
            'type' => $type,
            'data' => $data,
        ]);
    }
}
```

### Model: Opportunity.php
```php
<?php
namespace App\Models;

use App\Traits\BelongsToTenant;
use App\Events\{OpportunityStageChanged, OpportunityWon, OpportunityLost};
use Illuminate\Database\Eloquent\{Model, SoftDeletes};
use Illuminate\Database\Eloquent\Concerns\HasUuids;

class Opportunity extends Model
{
    use HasUuids, SoftDeletes, BelongsToTenant;
    
    protected $guarded = ['id'];
    
    protected $casts = [
        'value' => 'decimal:3',
        'expected_value' => 'decimal:3',
        'expected_close_date' => 'date',
        'won_at' => 'datetime',
        'lost_at' => 'datetime',
        'custom_fields' => 'array',
    ];
    
    protected static function booted(): void
    {
        static::saving(function ($opp) {
            $opp->expected_value = $opp->value * ($opp->probability / 100);
        });
    }
    
    // Relationships
    public function contact() { return $this->belongsTo(Contact::class); }
    public function pipeline() { return $this->belongsTo(Pipeline::class); }
    public function stage() { return $this->belongsTo(PipelineStage::class, 'stage_id'); }
    public function owner() { return $this->belongsTo(User::class, 'owner_id'); }
    public function products() { return $this->belongsToMany(Product::class, 'opportunity_products')->withPivot('quantity', 'price'); }
    public function activities() { return $this->morphMany(Activity::class, 'subject'); }
    
    // Scopes
    public function scopeOpen($query) { return $query->where('status', 'open'); }
    public function scopeWon($query) { return $query->where('status', 'won'); }
    public function scopeLost($query) { return $query->where('status', 'lost'); }
    
    // Methods
    public function moveToStage(PipelineStage $stage): void
    {
        $oldStage = $this->stage;
        $this->update([
            'stage_id' => $stage->id,
            'probability' => $stage->probability,
            'days_in_stage' => 0,
        ]);
        
        event(new OpportunityStageChanged($this, $oldStage, $stage));
    }
    
    public function markWon(): void
    {
        $this->update([
            'status' => 'won',
            'probability' => 100,
            'won_at' => now(),
        ]);
        
        $this->contact->convertToCustomer();
        event(new OpportunityWon($this));
    }
    
    public function markLost(string $reason = null): void
    {
        $this->update([
            'status' => 'lost',
            'probability' => 0,
            'lost_reason' => $reason,
            'lost_at' => now(),
        ]);
        
        event(new OpportunityLost($this));
    }
}
```

### Controller: ContactController.php
```php
<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Contact;
use App\Services\{ContactService, ContactImportService};
use App\Http\Requests\Contact\{StoreContactRequest, UpdateContactRequest, ImportContactsRequest, MergeContactsRequest};
use Illuminate\Http\{Request, JsonResponse};

class ContactController extends Controller
{
    public function __construct(
        private ContactService $contactService,
        private ContactImportService $importService
    ) {}
    
    public function index(Request $request): JsonResponse
    {
        $contacts = Contact::query()
            ->with(['owner', 'tags'])
            ->when($request->search, fn($q, $s) => $q->search($s))
            ->when($request->type, fn($q, $t) => $q->where('type', $t))
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->owner_id, fn($q, $o) => $q->where('owner_id', $o))
            ->when($request->tag_id, fn($q, $t) => $q->whereHas('tags', fn($q) => $q->where('tags.id', $t)))
            ->orderBy($request->sort_by ?? 'created_at', $request->sort_dir ?? 'desc')
            ->paginate($request->per_page ?? 25);
        
        return response()->json([
            'success' => true,
            'data' => $contacts,
        ]);
    }
    
    public function store(StoreContactRequest $request): JsonResponse
    {
        $contact = $this->contactService->create($request->validated());
        
        return response()->json([
            'success' => true,
            'message' => 'Contact created successfully',
            'data' => $contact->load(['owner', 'tags']),
        ], 201);
    }
    
    public function show(Contact $contact): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => $contact->load([
                'owner', 'tags', 'company', 'activities' => fn($q) => $q->latest()->limit(10),
                'opportunities' => fn($q) => $q->with('stage')->latest()->limit(5),
            ]),
        ]);
    }
    
    public function update(UpdateContactRequest $request, Contact $contact): JsonResponse
    {
        $contact = $this->contactService->update($contact, $request->validated());
        
        return response()->json([
            'success' => true,
            'message' => 'Contact updated successfully',
            'data' => $contact->fresh(['owner', 'tags']),
        ]);
    }
    
    public function destroy(Contact $contact): JsonResponse
    {
        $contact->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Contact deleted successfully',
        ]);
    }
    
    public function import(ImportContactsRequest $request): JsonResponse
    {
        $result = $this->importService->import($request->file('file'), $request->mapping);
        
        return response()->json([
            'success' => true,
            'message' => "Imported {$result['imported']} contacts",
            'data' => $result,
        ]);
    }
    
    public function export(Request $request): JsonResponse
    {
        $path = $this->importService->export($request->all());
        
        return response()->json([
            'success' => true,
            'data' => ['download_url' => $path],
        ]);
    }
    
    public function merge(MergeContactsRequest $request): JsonResponse
    {
        $primary = $this->contactService->merge(
            $request->primary_id,
            $request->secondary_ids
        );
        
        return response()->json([
            'success' => true,
            'message' => 'Contacts merged successfully',
            'data' => $primary,
        ]);
    }
    
    public function timeline(Contact $contact): JsonResponse
    {
        $activities = $contact->activities()
            ->with('user')
            ->latest()
            ->paginate(20);
        
        return response()->json([
            'success' => true,
            'data' => $activities,
        ]);
    }
    
    public function addTags(Request $request, Contact $contact): JsonResponse
    {
        $contact->tags()->syncWithoutDetaching($request->tag_ids);
        
        return response()->json([
            'success' => true,
            'data' => $contact->fresh('tags'),
        ]);
    }
    
    public function convert(Contact $contact): JsonResponse
    {
        $contact->convertToCustomer();
        
        return response()->json([
            'success' => true,
            'message' => 'Contact converted to customer',
            'data' => $contact->fresh(),
        ]);
    }
}
```

### Controller: OpportunityController.php
```php
<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\{Opportunity, Pipeline, PipelineStage};
use App\Services\OpportunityService;
use App\Http\Requests\Opportunity\{StoreOpportunityRequest, UpdateOpportunityRequest, MoveOpportunityRequest};
use Illuminate\Http\{Request, JsonResponse};

class OpportunityController extends Controller
{
    public function __construct(private OpportunityService $service) {}
    
    public function index(Request $request): JsonResponse
    {
        $opps = Opportunity::query()
            ->with(['contact', 'stage', 'owner'])
            ->when($request->pipeline_id, fn($q, $p) => $q->where('pipeline_id', $p))
            ->when($request->stage_id, fn($q, $s) => $q->where('stage_id', $s))
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->owner_id, fn($q, $o) => $q->where('owner_id', $o))
            ->when($request->search, fn($q, $s) => $q->where('name', 'like', "%{$s}%"))
            ->orderBy($request->sort_by ?? 'created_at', $request->sort_dir ?? 'desc')
            ->paginate($request->per_page ?? 25);
        
        return response()->json(['success' => true, 'data' => $opps]);
    }
    
    public function store(StoreOpportunityRequest $request): JsonResponse
    {
        $opp = $this->service->create($request->validated());
        
        return response()->json([
            'success' => true,
            'message' => 'Opportunity created',
            'data' => $opp->load(['contact', 'stage', 'owner']),
        ], 201);
    }
    
    public function show(Opportunity $opportunity): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => $opportunity->load(['contact', 'pipeline', 'stage', 'owner', 'products', 'activities']),
        ]);
    }
    
    public function update(UpdateOpportunityRequest $request, Opportunity $opportunity): JsonResponse
    {
        $opportunity = $this->service->update($opportunity, $request->validated());
        
        return response()->json([
            'success' => true,
            'message' => 'Opportunity updated',
            'data' => $opportunity->fresh(['contact', 'stage', 'owner']),
        ]);
    }
    
    public function destroy(Opportunity $opportunity): JsonResponse
    {
        $opportunity->delete();
        return response()->json(['success' => true, 'message' => 'Opportunity deleted']);
    }
    
    public function kanban(Pipeline $pipeline): JsonResponse
    {
        $stages = $pipeline->stages()
            ->with(['opportunities' => fn($q) => $q->with('contact', 'owner')->orderBy('position')])
            ->orderBy('position')
            ->get();
        
        return response()->json([
            'success' => true,
            'data' => [
                'pipeline' => $pipeline,
                'stages' => $stages,
            ],
        ]);
    }
    
    public function move(MoveOpportunityRequest $request, Opportunity $opportunity): JsonResponse
    {
        $stage = PipelineStage::findOrFail($request->stage_id);
        $opportunity->moveToStage($stage);
        
        if ($request->has('position')) {
            $opportunity->update(['position' => $request->position]);
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Opportunity moved',
            'data' => $opportunity->fresh(['stage']),
        ]);
    }
    
    public function markWon(Opportunity $opportunity): JsonResponse
    {
        $opportunity->markWon();
        
        return response()->json([
            'success' => true,
            'message' => 'Opportunity marked as won',
            'data' => $opportunity->fresh(),
        ]);
    }
    
    public function markLost(Request $request, Opportunity $opportunity): JsonResponse
    {
        $opportunity->markLost($request->reason);
        
        return response()->json([
            'success' => true,
            'message' => 'Opportunity marked as lost',
            'data' => $opportunity->fresh(),
        ]);
    }
}
```

### Routes: routes/api/crm.php
```php
<?php
use App\Http\Controllers\Api\{
    ContactController,
    TagController,
    PipelineController,
    OpportunityController,
    ActivityController,
    ProductController,
    SmartListController,
    CrmDashboardController
};
use Illuminate\Support\Facades\Route;

Route::middleware('auth:sanctum')->group(function () {
    // Contacts
    Route::apiResource('contacts', ContactController::class);
    Route::post('contacts/import', [ContactController::class, 'import']);
    Route::get('contacts/export', [ContactController::class, 'export']);
    Route::post('contacts/merge', [ContactController::class, 'merge']);
    Route::get('contacts/{contact}/timeline', [ContactController::class, 'timeline']);
    Route::post('contacts/{contact}/tags', [ContactController::class, 'addTags']);
    Route::delete('contacts/{contact}/tags/{tag}', [ContactController::class, 'removeTag']);
    Route::post('contacts/{contact}/convert', [ContactController::class, 'convert']);
    
    // Tags
    Route::apiResource('tags', TagController::class);
    
    // Pipelines
    Route::apiResource('pipelines', PipelineController::class);
    Route::post('pipelines/{pipeline}/stages/reorder', [PipelineController::class, 'reorderStages']);
    Route::apiResource('pipelines.stages', PipelineStageController::class)->shallow();
    
    // Opportunities
    Route::apiResource('opportunities', OpportunityController::class);
    Route::get('pipelines/{pipeline}/kanban', [OpportunityController::class, 'kanban']);
    Route::post('opportunities/{opportunity}/move', [OpportunityController::class, 'move']);
    Route::post('opportunities/{opportunity}/won', [OpportunityController::class, 'markWon']);
    Route::post('opportunities/{opportunity}/lost', [OpportunityController::class, 'markLost']);
    
    // Activities
    Route::apiResource('activities', ActivityController::class);
    Route::post('activities/{activity}/complete', [ActivityController::class, 'complete']);
    Route::get('activities/upcoming', [ActivityController::class, 'upcoming']);
    
    // Products
    Route::apiResource('products', ProductController::class);
    
    // Smart Lists
    Route::apiResource('smart-lists', SmartListController::class);
    Route::get('smart-lists/{smartList}/contacts', [SmartListController::class, 'contacts']);
    
    // Dashboard
    Route::get('crm/dashboard', [CrmDashboardController::class, 'index']);
    Route::get('crm/reports/pipeline', [CrmDashboardController::class, 'pipelineReport']);
});
```

---

## ✅ COMPLETION CHECKLIST

- [ ] All 8 migrations created
- [ ] All 10 models with relationships
- [ ] All 8 controllers with CRUD
- [ ] All 6 services
- [ ] All 14 request validations
- [ ] Routes file complete
- [ ] 4 jobs for background tasks
- [ ] 5 events for real-time updates
- [ ] 3 seeders with sample data
- [ ] 6 feature tests

---

## 🔗 DEPENDENCIES
- Bot 1: Tenant, User models, BelongsToTenant trait

## 🔗 OTHER BOTS DEPEND ON
- Bot 3, 4: Contact model for invoices/bills
- Bot 5: Contact model for project clients
- Bot 7: Contact model for communication
- Bot 8: Contact model for campaigns

---

## 📤 OUTPUT
Save all files to: `/home/claude/business-platform/`
When done, create: `/home/claude/business-platform/docs/PROGRESS_BOT_2.md`
