<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Tags
        Schema::create('tags', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('color', 7)->default('#3B82F6');
            $table->string('type')->default('contact');
            $table->timestamps();
            
            $table->unique(['tenant_id', 'name', 'type']);
        });

        // Contacts
        Schema::create('contacts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('owner_id')->nullable()->constrained('users')->nullOnDelete();
            $table->enum('type', ['person', 'company'])->default('person');
            $table->foreignId('company_id')->nullable()->constrained('contacts')->nullOnDelete();
            
            $table->string('first_name')->nullable();
            $table->string('last_name')->nullable();
            $table->string('company_name')->nullable();
            $table->string('job_title')->nullable();
            
            $table->string('email')->nullable();
            $table->string('phone')->nullable();
            $table->string('mobile')->nullable();
            $table->string('whatsapp')->nullable();
            $table->string('website')->nullable();
            
            $table->text('address')->nullable();
            $table->string('city')->nullable();
            $table->string('state')->nullable();
            $table->string('postal_code')->nullable();
            $table->string('country', 2)->nullable();
            
            $table->string('linkedin')->nullable();
            $table->string('twitter')->nullable();
            $table->string('facebook')->nullable();
            
            $table->string('cr_number')->nullable();
            $table->string('nationality', 2)->nullable();
            $table->enum('ownership_type', ['100_foreign', '51_bahraini', 'gcc_owned', 'mixed'])->nullable();
            $table->string('business_sector')->nullable();
            
            $table->string('source')->nullable();
            $table->enum('lead_status', ['new', 'contacted', 'qualified', 'proposal', 'negotiation', 'won', 'lost'])->nullable();
            $table->decimal('lead_score', 5, 2)->default(0);
            
            $table->string('currency', 3)->default('BHD');
            $table->decimal('lifetime_value', 15, 2)->default(0);
            
            $table->date('date_of_birth')->nullable();
            $table->timestamp('last_contacted_at')->nullable();
            $table->timestamp('converted_at')->nullable();
            
            $table->enum('status', ['active', 'inactive', 'archived'])->default('active');
            $table->boolean('do_not_contact')->default(false);
            $table->boolean('email_opt_out')->default(false);
            $table->boolean('sms_opt_out')->default(false);
            
            $table->json('custom_fields')->nullable();
            $table->text('notes')->nullable();
            
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['tenant_id', 'email']);
            $table->index(['tenant_id', 'phone']);
            $table->index(['tenant_id', 'status']);
            $table->index(['tenant_id', 'lead_status']);
            $table->index(['tenant_id', 'owner_id']);
        });

        Schema::create('contact_tag', function (Blueprint $table) {
            $table->foreignId('contact_id')->constrained()->cascadeOnDelete();
            $table->foreignId('tag_id')->constrained()->cascadeOnDelete();
            $table->primary(['contact_id', 'tag_id']);
        });

        // Custom Fields
        Schema::create('custom_field_definitions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('entity_type');
            $table->string('name');
            $table->string('label');
            $table->enum('type', ['text', 'textarea', 'number', 'decimal', 'date', 'datetime', 'select', 'multiselect', 'checkbox', 'url', 'email', 'phone', 'user']);
            $table->json('options')->nullable();
            $table->boolean('required')->default(false);
            $table->string('default_value')->nullable();
            $table->integer('position')->default(0);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            
            $table->unique(['tenant_id', 'entity_type', 'name']);
        });

        // Pipelines
        Schema::create('pipelines', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->text('description')->nullable();
            $table->string('type')->default('sales');
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->integer('position')->default(0);
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('pipeline_stages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('pipeline_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('color', 7)->default('#3B82F6');
            $table->integer('position')->default(0);
            $table->decimal('probability', 5, 2)->default(0);
            $table->boolean('is_won')->default(false);
            $table->boolean('is_lost')->default(false);
            $table->integer('rotting_days')->nullable();
            $table->timestamps();
        });

        // Opportunities
        Schema::create('opportunities', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('pipeline_id')->constrained()->cascadeOnDelete();
            $table->foreignId('stage_id')->constrained('pipeline_stages')->cascadeOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('company_id')->nullable()->constrained('contacts')->nullOnDelete();
            $table->foreignId('owner_id')->nullable()->constrained('users')->nullOnDelete();
            
            $table->string('title');
            $table->text('description')->nullable();
            $table->decimal('value', 15, 2)->default(0);
            $table->string('currency', 3)->default('BHD');
            $table->date('expected_close_date')->nullable();
            $table->date('actual_close_date')->nullable();
            $table->decimal('probability', 5, 2)->nullable();
            $table->string('source')->nullable();
            $table->enum('status', ['open', 'won', 'lost'])->default('open');
            $table->text('lost_reason')->nullable();
            
            $table->enum('package', ['standard', 'gold', 'premium'])->nullable();
            $table->string('business_activity')->nullable();
            $table->string('legal_form')->nullable();
            
            $table->integer('position')->default(0);
            $table->json('custom_fields')->nullable();
            
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['tenant_id', 'pipeline_id', 'stage_id']);
            $table->index(['tenant_id', 'status']);
            $table->index(['tenant_id', 'owner_id']);
        });

        // Activities
        Schema::create('activities', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->morphs('activitable');
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            
            $table->enum('type', ['call', 'email', 'meeting', 'task', 'note', 'sms', 'whatsapp', 'stage_change', 'system']);
            $table->string('title');
            $table->text('description')->nullable();
            $table->timestamp('scheduled_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->integer('duration_minutes')->nullable();
            $table->enum('outcome', ['completed', 'no_answer', 'rescheduled', 'cancelled'])->nullable();
            $table->json('metadata')->nullable();
            
            $table->timestamps();
            
            $table->index(['tenant_id', 'type']);
        });

        // Products/Services
        Schema::create('products', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('sku')->nullable();
            $table->text('description')->nullable();
            $table->enum('type', ['product', 'service'])->default('service');
            $table->decimal('price', 15, 2)->default(0);
            $table->decimal('cost', 15, 2)->default(0);
            $table->string('currency', 3)->default('BHD');
            $table->decimal('tax_rate', 5, 2)->default(10);
            $table->string('unit')->default('unit');
            $table->boolean('is_active')->default(true);
            $table->integer('stock_quantity')->nullable();
            $table->integer('low_stock_threshold')->nullable();
            $table->string('category')->nullable();
            $table->json('custom_fields')->nullable();
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['tenant_id', 'sku']);
        });

        Schema::create('opportunity_products', function (Blueprint $table) {
            $table->id();
            $table->foreignId('opportunity_id')->constrained()->cascadeOnDelete();
            $table->foreignId('product_id')->nullable()->constrained()->nullOnDelete();
            $table->string('name');
            $table->text('description')->nullable();
            $table->decimal('quantity', 10, 2)->default(1);
            $table->decimal('unit_price', 15, 2)->default(0);
            $table->decimal('discount', 5, 2)->default(0);
            $table->decimal('tax_rate', 5, 2)->default(10);
            $table->decimal('total', 15, 2)->default(0);
            $table->timestamps();
        });

        // Smart Lists
        Schema::create('smart_lists', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('entity_type');
            $table->json('filters');
            $table->json('columns')->nullable();
            $table->string('sort_field')->nullable();
            $table->string('sort_direction')->default('asc');
            $table->boolean('is_shared')->default(false);
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('smart_lists');
        Schema::dropIfExists('opportunity_products');
        Schema::dropIfExists('products');
        Schema::dropIfExists('activities');
        Schema::dropIfExists('opportunities');
        Schema::dropIfExists('pipeline_stages');
        Schema::dropIfExists('pipelines');
        Schema::dropIfExists('custom_field_definitions');
        Schema::dropIfExists('contact_tag');
        Schema::dropIfExists('contacts');
        Schema::dropIfExists('tags');
    }
};
