<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Email Accounts
        Schema::create('email_accounts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->string('name');
            $table->string('email');
            $table->enum('provider', ['smtp', 'sendgrid', 'mailgun', 'ses', 'gmail', 'outlook']);
            $table->json('settings')->nullable();
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Phone Numbers
        Schema::create('phone_numbers', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('number');
            $table->string('friendly_name')->nullable();
            $table->enum('provider', ['twilio', 'vonage', 'plivo']);
            $table->json('capabilities'); // ['sms', 'mms', 'voice', 'whatsapp']
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Conversations (Unified Inbox)
        Schema::create('conversations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('assigned_to')->nullable()->constrained('users')->nullOnDelete();
            
            $table->enum('channel', ['email', 'sms', 'whatsapp', 'phone', 'chat', 'facebook', 'instagram']);
            $table->string('channel_id')->nullable();
            $table->string('subject')->nullable();
            
            $table->enum('status', ['open', 'pending', 'resolved', 'closed'])->default('open');
            $table->enum('priority', ['low', 'normal', 'high', 'urgent'])->default('normal');
            
            $table->timestamp('last_message_at')->nullable();
            $table->integer('message_count')->default(0);
            $table->boolean('is_read')->default(false);
            $table->boolean('is_starred')->default(false);
            
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['tenant_id', 'contact_id']);
            $table->index(['tenant_id', 'channel']);
            $table->index(['tenant_id', 'status']);
            $table->index(['tenant_id', 'assigned_to']);
        });

        // Messages
        Schema::create('messages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('conversation_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            
            $table->enum('direction', ['inbound', 'outbound']);
            $table->enum('channel', ['email', 'sms', 'whatsapp', 'phone', 'chat']);
            
            $table->string('from')->nullable();
            $table->string('to')->nullable();
            $table->string('subject')->nullable();
            $table->longText('body')->nullable();
            $table->text('body_plain')->nullable();
            
            $table->enum('status', ['queued', 'sending', 'sent', 'delivered', 'read', 'failed', 'bounced'])->default('queued');
            $table->text('error_message')->nullable();
            
            $table->string('external_id')->nullable();
            $table->json('metadata')->nullable();
            
            $table->timestamp('sent_at')->nullable();
            $table->timestamp('delivered_at')->nullable();
            $table->timestamp('read_at')->nullable();
            
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['conversation_id', 'created_at']);
            $table->index(['tenant_id', 'channel', 'status']);
        });

        // Message Attachments
        Schema::create('message_attachments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('message_id')->constrained()->cascadeOnDelete();
            $table->string('filename');
            $table->string('original_filename');
            $table->string('path');
            $table->string('disk')->default('local');
            $table->string('mime_type');
            $table->unsignedBigInteger('size');
            $table->string('url')->nullable();
            $table->timestamps();
        });

        // Email Templates
        Schema::create('email_templates', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('subject');
            $table->longText('body');
            $table->enum('type', ['email', 'sms', 'whatsapp'])->default('email');
            $table->string('category')->nullable();
            $table->json('variables')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            $table->softDeletes();
        });

        // Campaigns
        Schema::create('campaigns', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('created_by')->constrained('users')->cascadeOnDelete();
            
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('type', ['email', 'sms', 'whatsapp']);
            $table->enum('status', ['draft', 'scheduled', 'sending', 'sent', 'paused', 'cancelled'])->default('draft');
            
            $table->foreignId('template_id')->nullable()->constrained('email_templates')->nullOnDelete();
            $table->string('subject')->nullable();
            $table->longText('content')->nullable();
            
            $table->foreignId('smart_list_id')->nullable()->constrained('smart_lists')->nullOnDelete();
            $table->json('recipient_filters')->nullable();
            $table->integer('recipient_count')->default(0);
            
            $table->timestamp('scheduled_at')->nullable();
            $table->timestamp('started_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            
            $table->integer('sent_count')->default(0);
            $table->integer('delivered_count')->default(0);
            $table->integer('opened_count')->default(0);
            $table->integer('clicked_count')->default(0);
            $table->integer('bounced_count')->default(0);
            $table->integer('unsubscribed_count')->default(0);
            
            $table->timestamps();
            $table->softDeletes();
        });

        // Campaign Recipients
        Schema::create('campaign_recipients', function (Blueprint $table) {
            $table->id();
            $table->foreignId('campaign_id')->constrained()->cascadeOnDelete();
            $table->foreignId('contact_id')->constrained()->cascadeOnDelete();
            $table->enum('status', ['pending', 'sent', 'delivered', 'opened', 'clicked', 'bounced', 'failed', 'unsubscribed'])->default('pending');
            $table->timestamp('sent_at')->nullable();
            $table->timestamp('delivered_at')->nullable();
            $table->timestamp('opened_at')->nullable();
            $table->timestamp('clicked_at')->nullable();
            $table->text('error_message')->nullable();
            $table->timestamps();
            
            $table->unique(['campaign_id', 'contact_id']);
        });

        // Automation Workflows
        Schema::create('workflows', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('created_by')->constrained('users')->cascadeOnDelete();
            
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('status', ['draft', 'active', 'paused', 'archived'])->default('draft');
            
            $table->json('trigger'); // {type: 'tag_added', conditions: {...}}
            $table->json('steps'); // [{type: 'send_email', delay: '1d', params: {...}}, ...]
            
            $table->integer('enrolled_count')->default(0);
            $table->integer('completed_count')->default(0);
            
            $table->timestamps();
            $table->softDeletes();
        });

        // Workflow Enrollments
        Schema::create('workflow_enrollments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('workflow_id')->constrained()->cascadeOnDelete();
            $table->foreignId('contact_id')->constrained()->cascadeOnDelete();
            $table->integer('current_step')->default(0);
            $table->enum('status', ['active', 'completed', 'stopped', 'failed'])->default('active');
            $table->timestamp('next_action_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->json('step_history')->nullable();
            $table->timestamps();
            
            $table->index(['workflow_id', 'status']);
        });

        // Call Logs
        Schema::create('call_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('conversation_id')->nullable()->constrained()->nullOnDelete();
            
            $table->string('from_number');
            $table->string('to_number');
            $table->enum('direction', ['inbound', 'outbound']);
            $table->enum('status', ['queued', 'ringing', 'in_progress', 'completed', 'failed', 'busy', 'no_answer', 'cancelled']);
            
            $table->integer('duration_seconds')->default(0);
            $table->string('recording_url')->nullable();
            $table->text('transcription')->nullable();
            
            $table->string('external_id')->nullable();
            $table->json('metadata')->nullable();
            
            $table->timestamp('started_at')->nullable();
            $table->timestamp('answered_at')->nullable();
            $table->timestamp('ended_at')->nullable();
            
            $table->timestamps();
            
            $table->index(['tenant_id', 'contact_id']);
        });

        // Calendar/Appointments
        Schema::create('calendars', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->string('name');
            $table->string('color', 7)->default('#3B82F6');
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->json('settings')->nullable();
            $table->timestamps();
        });

        Schema::create('appointments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('calendar_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            
            $table->string('title');
            $table->text('description')->nullable();
            $table->string('location')->nullable();
            $table->string('meeting_url')->nullable();
            
            $table->timestamp('start_time');
            $table->timestamp('end_time');
            $table->boolean('all_day')->default(false);
            $table->string('timezone')->default('Asia/Bahrain');
            
            $table->enum('status', ['scheduled', 'confirmed', 'completed', 'cancelled', 'no_show'])->default('scheduled');
            
            $table->string('recurrence_rule')->nullable();
            $table->foreignId('recurrence_parent_id')->nullable()->constrained('appointments')->cascadeOnDelete();
            
            $table->integer('reminder_minutes')->default(30);
            $table->boolean('reminder_sent')->default(false);
            
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['tenant_id', 'calendar_id', 'start_time']);
            $table->index(['tenant_id', 'user_id', 'start_time']);
        });

        // Booking Links
        Schema::create('booking_links', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('calendar_id')->constrained()->cascadeOnDelete();
            
            $table->string('slug')->unique();
            $table->string('name');
            $table->text('description')->nullable();
            $table->integer('duration_minutes')->default(30);
            $table->json('availability'); // {mon: [{start: '09:00', end: '17:00'}], ...}
            $table->integer('buffer_before')->default(0);
            $table->integer('buffer_after')->default(0);
            $table->integer('min_notice_hours')->default(24);
            $table->integer('max_days_advance')->default(60);
            
            $table->boolean('is_active')->default(true);
            $table->json('questions')->nullable();
            $table->json('settings')->nullable();
            
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('booking_links');
        Schema::dropIfExists('appointments');
        Schema::dropIfExists('calendars');
        Schema::dropIfExists('call_logs');
        Schema::dropIfExists('workflow_enrollments');
        Schema::dropIfExists('workflows');
        Schema::dropIfExists('campaign_recipients');
        Schema::dropIfExists('campaigns');
        Schema::dropIfExists('email_templates');
        Schema::dropIfExists('message_attachments');
        Schema::dropIfExists('messages');
        Schema::dropIfExists('conversations');
        Schema::dropIfExists('phone_numbers');
        Schema::dropIfExists('email_accounts');
    }
};
