<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('bank_accounts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('account_id')->nullable()->constrained('chart_of_accounts')->nullOnDelete();
            $table->string('name');
            $table->string('bank_name')->nullable();
            $table->string('account_number')->nullable();
            $table->string('iban')->nullable();
            $table->string('swift_code')->nullable();
            $table->string('currency', 3)->default('BHD');
            $table->decimal('opening_balance', 15, 3)->default(0);
            $table->decimal('current_balance', 15, 3)->default(0);
            $table->string('plaid_access_token')->nullable();
            $table->string('plaid_item_id')->nullable();
            $table->string('plaid_account_id')->nullable();
            $table->string('plaid_cursor')->nullable();
            $table->timestamp('last_synced_at')->nullable();
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamps();
            $table->softDeletes();
            $table->index(['tenant_id', 'is_active']);
        });

        Schema::create('bank_transactions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('bank_account_id')->constrained()->cascadeOnDelete();
            $table->foreignId('account_id')->nullable()->constrained('chart_of_accounts')->nullOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            $table->string('provider_id')->nullable();
            $table->date('transaction_date');
            $table->string('description');
            $table->decimal('amount', 15, 3);
            $table->string('type'); // credit, debit
            $table->string('category')->nullable();
            $table->string('subcategory')->nullable();
            $table->string('merchant_name')->nullable();
            $table->boolean('pending')->default(false);
            $table->string('status')->default('unmatched');
            $table->foreignId('matched_invoice_id')->nullable()->constrained('invoices')->nullOnDelete();
            $table->foreignId('matched_bill_id')->nullable()->constrained('bills')->nullOnDelete();
            $table->foreignId('matched_payment_id')->nullable()->constrained('payments')->nullOnDelete();
            $table->boolean('is_reconciled')->default(false);
            $table->foreignId('reconciliation_id')->nullable();
            $table->json('raw_data')->nullable();
            $table->timestamps();
            $table->softDeletes();
            $table->index(['tenant_id', 'transaction_date']);
            $table->index(['tenant_id', 'status']);
            $table->index('provider_id');
        });

        Schema::create('bank_reconciliations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('bank_account_id')->constrained()->cascadeOnDelete();
            $table->date('statement_date');
            $table->decimal('statement_balance', 15, 3);
            $table->decimal('ledger_balance', 15, 3);
            $table->decimal('difference', 15, 3)->default(0);
            $table->string('status')->default('draft');
            $table->timestamp('reconciled_at')->nullable();
            $table->foreignId('reconciled_by')->nullable()->constrained('users')->nullOnDelete();
            $table->text('notes')->nullable();
            $table->foreignId('created_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamps();
            $table->index(['tenant_id', 'bank_account_id']);
        });

        Schema::create('bank_rules', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('bank_account_id')->nullable()->constrained()->nullOnDelete();
            $table->string('name');
            $table->json('conditions');
            $table->foreignId('account_id')->nullable()->constrained('chart_of_accounts')->nullOnDelete();
            $table->foreignId('contact_id')->nullable()->constrained()->nullOnDelete();
            $table->string('transaction_type')->nullable();
            $table->boolean('auto_categorize')->default(true);
            $table->boolean('is_active')->default(true);
            $table->integer('priority')->default(0);
            $table->timestamps();
            $table->index(['tenant_id', 'is_active']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('bank_rules');
        Schema::dropIfExists('bank_reconciliations');
        Schema::dropIfExists('bank_transactions');
        Schema::dropIfExists('bank_accounts');
    }
};
