# 🤖 BOT 10: FRONTEND - ACCOUNTING / TASKS

## 📋 YOUR RESPONSIBILITY
Build Vue.js/React frontend for accounting (invoices, bills, reports) and task management (projects, tasks, time tracking).

---

## 🛠️ TECH STACK
Same as Bot 9 (Vue 3 or React 18 + Tailwind)

---

## 📁 FILES TO CREATE

### 1. PAGES - INVOICES (8 files)
```
resources/js/Pages/Invoices/
├── Index.vue                # Invoice list
├── Show.vue                 # Invoice detail
├── Create.vue               # Create invoice
├── Edit.vue                 # Edit invoice
├── Preview.vue              # PDF preview
├── Partials/
│   ├── InvoiceTable.vue
│   ├── InvoiceForm.vue
│   ├── InvoiceItems.vue
│   ├── InvoicePayments.vue
│   └── InvoiceActions.vue
```

### 2. PAGES - ESTIMATES (6 files)
```
resources/js/Pages/Estimates/
├── Index.vue
├── Show.vue
├── Create.vue
├── Edit.vue
├── Partials/
│   ├── EstimateForm.vue
│   └── EstimateItems.vue
```

### 3. PAGES - BILLS & EXPENSES (8 files)
```
resources/js/Pages/Bills/
├── Index.vue
├── Show.vue
├── Create.vue
├── Edit.vue
├── Partials/
│   ├── BillForm.vue
│   └── BillItems.vue

resources/js/Pages/Expenses/
├── Index.vue
├── Create.vue
├── Partials/
│   ├── ExpenseForm.vue
│   └── ExpenseReceipt.vue
```

### 4. PAGES - BANKING (6 files)
```
resources/js/Pages/Banking/
├── Index.vue                # Bank accounts list
├── Transactions.vue         # Transaction list
├── Reconciliation.vue       # Bank reconciliation
├── Partials/
│   ├── TransactionTable.vue
│   ├── TransactionMatch.vue
│   ├── PlaidConnect.vue
│   └── ReconciliationWizard.vue
```

### 5. PAGES - ACCOUNTING REPORTS (6 files)
```
resources/js/Pages/Reports/
├── Index.vue                # Reports hub
├── ProfitLoss.vue          # Income statement
├── BalanceSheet.vue
├── CashFlow.vue
├── TrialBalance.vue
├── AccountsAging.vue        # AR/AP aging
```

### 6. PAGES - PROJECTS (8 files)
```
resources/js/Pages/Projects/
├── Index.vue                # Project list
├── Show.vue                 # Project detail
├── Create.vue
├── Edit.vue
├── Kanban.vue               # Task kanban
├── Gantt.vue                # Gantt chart
├── Partials/
│   ├── ProjectForm.vue
│   ├── ProjectMembers.vue
│   ├── ProjectStats.vue
│   └── ProjectSettings.vue
```

### 7. PAGES - TASKS (8 files)
```
resources/js/Pages/Tasks/
├── Index.vue                # All tasks
├── MyTasks.vue              # My tasks
├── Show.vue                 # Task detail
├── Partials/
│   ├── TaskTable.vue
│   ├── TaskCard.vue
│   ├── TaskForm.vue
│   ├── TaskComments.vue
│   ├── TaskChecklist.vue
│   ├── TaskAttachments.vue
│   └── TaskTimeEntries.vue
```

### 8. PAGES - TIME TRACKING (6 files)
```
resources/js/Pages/Time/
├── Index.vue                # Time entries
├── Timer.vue                # Active timer
├── Timesheet.vue            # Weekly timesheet
├── Partials/
│   ├── TimeEntryForm.vue
│   ├── TimerWidget.vue
│   ├── TimesheetGrid.vue
│   └── TimeReport.vue
```

### 9. PAGES - SETTINGS (8 files)
```
resources/js/Pages/Settings/
├── Index.vue                # Settings overview
├── Profile.vue              # User profile
├── Company.vue              # Company/tenant settings
├── Billing.vue              # Subscription
├── Team.vue                 # Team members
├── Integrations.vue         # Third-party integrations
├── Accounting.vue           # Chart of accounts, tax rates
├── Notifications.vue        # Notification preferences
```

### 10. COMPONENTS - ACCOUNTING (10 files)
```
resources/js/Components/Accounting/
├── InvoicePreview.vue
├── LineItemsEditor.vue
├── TaxSelector.vue
├── CurrencyDisplay.vue
├── PaymentBadge.vue
├── AccountSelector.vue
├── AmountInput.vue
├── DateRangePicker.vue
├── ReportChart.vue
└── ReportTable.vue
```

### 11. COMPONENTS - TASKS (10 files)
```
resources/js/Components/Tasks/
├── KanbanBoard.vue
├── KanbanColumn.vue
├── TaskCard.vue
├── GanttChart.vue
├── GanttRow.vue
├── TimerButton.vue
├── TimeEntryRow.vue
├── PriorityBadge.vue
├── DueDatePicker.vue
└── AssigneeSelector.vue
```

### 12. COMPOSABLES/HOOKS (8 files)
```
resources/js/Composables/
├── useInvoices.js
├── useEstimates.js
├── useBills.js
├── useBanking.js
├── useProjects.js
├── useTasks.js
├── useTimer.js
└── useReports.js
```

### 13. STORES (8 files)
```
resources/js/Stores/
├── invoices.js
├── estimates.js
├── bills.js
├── banking.js
├── projects.js
├── tasks.js
├── timer.js
└── reports.js
```

### 14. API SERVICES (8 files)
```
resources/js/Services/
├── invoiceService.js
├── estimateService.js
├── billService.js
├── bankingService.js
├── projectService.js
├── taskService.js
├── timeService.js
└── reportService.js
```

---

## 📝 PAGE SPECIFICATIONS

### Invoices/Index.vue
```vue
<template>
  <AppLayout title="Invoices">
    <div class="flex justify-between mb-6">
      <div class="flex gap-4">
        <Select v-model="filters.status" :options="statusOptions" placeholder="All Statuses" />
        <DateRangePicker v-model="filters.dateRange" />
      </div>
      <Button @click="$router.push('/invoices/create')">New Invoice</Button>
    </div>
    
    <div class="grid grid-cols-4 gap-4 mb-6">
      <Card>
        <p class="text-sm text-gray-500">Total Invoiced</p>
        <p class="text-2xl font-bold">{{ formatCurrency(stats.total) }}</p>
      </Card>
      <Card>
        <p class="text-sm text-gray-500">Paid</p>
        <p class="text-2xl font-bold text-green-600">{{ formatCurrency(stats.paid) }}</p>
      </Card>
      <Card>
        <p class="text-sm text-gray-500">Outstanding</p>
        <p class="text-2xl font-bold text-yellow-600">{{ formatCurrency(stats.outstanding) }}</p>
      </Card>
      <Card>
        <p class="text-sm text-gray-500">Overdue</p>
        <p class="text-2xl font-bold text-red-600">{{ formatCurrency(stats.overdue) }}</p>
      </Card>
    </div>
    
    <InvoiceTable 
      :invoices="invoices" 
      :loading="loading"
      @view="viewInvoice"
      @edit="editInvoice"
      @send="sendInvoice"
      @delete="deleteInvoice"
    />
  </AppLayout>
</template>
```

### Invoices/Create.vue
```vue
<template>
  <AppLayout title="Create Invoice">
    <Card>
      <InvoiceForm v-model="invoice">
        <!-- Contact selector -->
        <ContactSelector v-model="invoice.contact_id" />
        
        <!-- Invoice details -->
        <div class="grid grid-cols-3 gap-4">
          <DatePicker v-model="invoice.invoice_date" label="Invoice Date" />
          <DatePicker v-model="invoice.due_date" label="Due Date" />
          <Select v-model="invoice.currency" :options="currencies" label="Currency" />
        </div>
        
        <!-- Line items -->
        <LineItemsEditor 
          v-model="invoice.items" 
          :taxRates="taxRates"
          @update:totals="updateTotals"
        />
        
        <!-- Totals -->
        <div class="w-64 ml-auto">
          <div class="flex justify-between py-2">
            <span>Subtotal:</span>
            <span>{{ formatCurrency(invoice.subtotal) }}</span>
          </div>
          <div class="flex justify-between py-2">
            <span>VAT (10%):</span>
            <span>{{ formatCurrency(invoice.tax_amount) }}</span>
          </div>
          <div class="flex justify-between py-2 font-bold border-t">
            <span>Total:</span>
            <span>{{ formatCurrency(invoice.total) }}</span>
          </div>
        </div>
        
        <!-- Notes -->
        <Textarea v-model="invoice.notes" label="Notes" />
      </InvoiceForm>
      
      <div class="flex justify-end gap-4 mt-6">
        <Button variant="secondary" @click="saveAsDraft">Save as Draft</Button>
        <Button @click="saveAndSend">Save & Send</Button>
      </div>
    </Card>
  </AppLayout>
</template>
```

### Projects/Kanban.vue
```vue
<template>
  <AppLayout :title="project.name">
    <div class="flex justify-between mb-6">
      <div class="flex gap-4">
        <Button variant="ghost" @click="view = 'kanban'" :active="view === 'kanban'">Kanban</Button>
        <Button variant="ghost" @click="view = 'list'" :active="view === 'list'">List</Button>
        <Button variant="ghost" @click="view = 'gantt'" :active="view === 'gantt'">Gantt</Button>
      </div>
      <Button @click="taskModal = true">Add Task</Button>
    </div>
    
    <KanbanBoard v-if="view === 'kanban'">
      <KanbanColumn
        v-for="status in statuses"
        :key="status.id"
        :status="status"
        :tasks="tasksByStatus[status.id]"
        @add="addTask(status.id)"
        @drop="moveTask"
      >
        <TaskCard
          v-for="task in tasksByStatus[status.id]"
          :key="task.id"
          :task="task"
          draggable
          @click="openTask(task)"
        />
      </KanbanColumn>
    </KanbanBoard>
    
    <GanttChart v-if="view === 'gantt'" :tasks="tasks" :milestones="milestones" />
    
    <TaskModal v-model="taskModal" :task="selectedTask" :project="project" @save="saveTask" />
  </AppLayout>
</template>
```

### Time/Timer.vue
```vue
<template>
  <AppLayout title="Time Tracking">
    <!-- Active Timer -->
    <Card v-if="activeTimer" class="mb-6 border-green-500">
      <div class="flex items-center justify-between">
        <div>
          <p class="font-medium">{{ activeTimer.task?.title || 'No task' }}</p>
          <p class="text-sm text-gray-500">{{ activeTimer.project?.name }}</p>
        </div>
        <div class="text-3xl font-mono font-bold">
          {{ formatDuration(elapsedTime) }}
        </div>
        <div class="flex gap-2">
          <Button @click="stopTimer" variant="danger">Stop</Button>
          <Button @click="discardTimer" variant="ghost">Discard</Button>
        </div>
      </div>
    </Card>
    
    <!-- Start New Timer -->
    <Card v-else class="mb-6">
      <div class="flex gap-4">
        <Select v-model="newTimer.project_id" :options="projects" placeholder="Select project" />
        <Select v-model="newTimer.task_id" :options="projectTasks" placeholder="Select task (optional)" />
        <Input v-model="newTimer.description" placeholder="What are you working on?" class="flex-1" />
        <Button @click="startTimer">Start Timer</Button>
      </div>
    </Card>
    
    <!-- Today's Entries -->
    <Card>
      <h2 class="text-lg font-medium mb-4">Today's Time</h2>
      <TimeEntryRow
        v-for="entry in todayEntries"
        :key="entry.id"
        :entry="entry"
        @edit="editEntry"
        @delete="deleteEntry"
      />
      <div class="flex justify-between mt-4 pt-4 border-t font-medium">
        <span>Total Today:</span>
        <span>{{ formatDuration(totalToday) }}</span>
      </div>
    </Card>
  </AppLayout>
</template>
```

### Reports/ProfitLoss.vue
```vue
<template>
  <AppLayout title="Profit & Loss Statement">
    <div class="flex justify-between mb-6">
      <DateRangePicker v-model="dateRange" :presets="['This Month', 'Last Month', 'This Quarter', 'This Year']" />
      <div class="flex gap-2">
        <Button variant="secondary" @click="exportPdf">Export PDF</Button>
        <Button variant="secondary" @click="exportExcel">Export Excel</Button>
      </div>
    </div>
    
    <Card>
      <h2 class="text-xl font-bold mb-4">Income Statement</h2>
      <p class="text-sm text-gray-500 mb-6">{{ formatDateRange(dateRange) }}</p>
      
      <!-- Revenue Section -->
      <ReportSection title="Revenue">
        <ReportRow v-for="account in revenue" :key="account.id" :account="account" />
        <ReportTotalRow label="Total Revenue" :amount="totalRevenue" />
      </ReportSection>
      
      <!-- Cost of Goods Sold -->
      <ReportSection title="Cost of Goods Sold">
        <ReportRow v-for="account in cogs" :key="account.id" :account="account" />
        <ReportTotalRow label="Total COGS" :amount="totalCogs" />
      </ReportSection>
      
      <!-- Gross Profit -->
      <ReportGrandTotal label="Gross Profit" :amount="grossProfit" class="bg-gray-50" />
      
      <!-- Operating Expenses -->
      <ReportSection title="Operating Expenses">
        <ReportRow v-for="account in expenses" :key="account.id" :account="account" />
        <ReportTotalRow label="Total Expenses" :amount="totalExpenses" />
      </ReportSection>
      
      <!-- Net Income -->
      <ReportGrandTotal 
        label="Net Income" 
        :amount="netIncome" 
        :class="netIncome >= 0 ? 'bg-green-50 text-green-800' : 'bg-red-50 text-red-800'" 
      />
    </Card>
  </AppLayout>
</template>
```

---

## 🔗 ROUTES

```js
// Add to resources/js/router/index.js

// Invoices
{ path: '/invoices', component: InvoicesIndex },
{ path: '/invoices/create', component: InvoicesCreate },
{ path: '/invoices/:id', component: InvoicesShow },
{ path: '/invoices/:id/edit', component: InvoicesEdit },

// Estimates
{ path: '/estimates', component: EstimatesIndex },
{ path: '/estimates/create', component: EstimatesCreate },
{ path: '/estimates/:id', component: EstimatesShow },

// Bills
{ path: '/bills', component: BillsIndex },
{ path: '/bills/create', component: BillsCreate },
{ path: '/bills/:id', component: BillsShow },

// Expenses
{ path: '/expenses', component: ExpensesIndex },
{ path: '/expenses/create', component: ExpensesCreate },

// Banking
{ path: '/banking', component: BankingIndex },
{ path: '/banking/:id/transactions', component: BankingTransactions },
{ path: '/banking/:id/reconcile', component: BankingReconciliation },

// Reports
{ path: '/reports', component: ReportsIndex },
{ path: '/reports/profit-loss', component: ReportsProfitLoss },
{ path: '/reports/balance-sheet', component: ReportsBalanceSheet },
{ path: '/reports/cash-flow', component: ReportsCashFlow },

// Projects
{ path: '/projects', component: ProjectsIndex },
{ path: '/projects/create', component: ProjectsCreate },
{ path: '/projects/:id', component: ProjectsShow },
{ path: '/projects/:id/kanban', component: ProjectsKanban },
{ path: '/projects/:id/gantt', component: ProjectsGantt },

// Tasks
{ path: '/tasks', component: TasksIndex },
{ path: '/my-tasks', component: TasksMyTasks },
{ path: '/tasks/:id', component: TasksShow },

// Time Tracking
{ path: '/time', component: TimeIndex },
{ path: '/timesheet', component: TimeTimesheet },

// Settings
{ path: '/settings', component: SettingsIndex },
{ path: '/settings/profile', component: SettingsProfile },
{ path: '/settings/company', component: SettingsCompany },
{ path: '/settings/team', component: SettingsTeam },
{ path: '/settings/billing', component: SettingsBilling },
{ path: '/settings/accounting', component: SettingsAccounting },
```

---

## ✅ COMPLETION CHECKLIST

- [ ] Invoice CRUD + line items
- [ ] Invoice PDF preview
- [ ] Estimate CRUD + convert
- [ ] Bill/Expense CRUD
- [ ] Bank accounts + transactions
- [ ] Bank reconciliation wizard
- [ ] Financial reports (P&L, Balance Sheet)
- [ ] Project CRUD + views
- [ ] Task kanban board
- [ ] Task detail + comments
- [ ] Gantt chart
- [ ] Time tracking + timer
- [ ] Timesheet view
- [ ] Settings pages
- [ ] All components
- [ ] Responsive design

---

## 📤 OUTPUT

Save to: `/home/claude/business-platform/resources/js/`
Update: `/home/claude/business-platform/docs/PROGRESS_BOT_10.md`
