# 🤖 BOT 1: CORE / AUTH / TENANTS MODULE

## 📋 YOUR RESPONSIBILITY
Build the authentication, user management, tenant management, and permission system.

---

## 📁 FILES TO CREATE

### 1. MODELS (8 files)
```
app/Models/
├── Tenant.php ✅ EXISTS - ENHANCE
├── User.php ✅ EXISTS - ENHANCE  
├── Team.php ✅ EXISTS - ENHANCE
├── TeamMember.php - CREATE
├── Invitation.php - CREATE
├── UserAvailability.php - CREATE
├── Permission.php - CREATE (or use Spatie)
├── Role.php - CREATE (or use Spatie)
```

### 2. CONTROLLERS (6 files)
```
app/Http/Controllers/Api/
├── AuthController.php - CREATE
├── TenantController.php - CREATE
├── UserController.php - CREATE
├── TeamController.php - CREATE
├── RoleController.php - CREATE
├── ProfileController.php - CREATE
```

### 3. SERVICES (4 files)
```
app/Services/
├── AuthService.php - CREATE
├── TenantService.php - CREATE
├── UserService.php - CREATE
├── InvitationService.php - CREATE
```

### 4. FORM REQUESTS (10 files)
```
app/Http/Requests/
├── Auth/LoginRequest.php
├── Auth/RegisterRequest.php
├── Auth/ForgotPasswordRequest.php
├── Auth/ResetPasswordRequest.php
├── Tenant/StoreTenantRequest.php
├── Tenant/UpdateTenantRequest.php
├── User/StoreUserRequest.php
├── User/UpdateUserRequest.php
├── Team/StoreTeamRequest.php
├── Invitation/StoreInvitationRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/auth.php - CREATE
```

### 6. JOBS (3 files)
```
app/Jobs/
├── SendInvitationEmail.php
├── ProcessTenantSetup.php
├── CleanupExpiredInvitations.php
```

### 7. NOTIFICATIONS (4 files)
```
app/Notifications/
├── WelcomeNotification.php
├── InvitationNotification.php
├── PasswordResetNotification.php
├── TwoFactorCodeNotification.php
```

### 8. POLICIES (3 files)
```
app/Policies/
├── TenantPolicy.php
├── UserPolicy.php
├── TeamPolicy.php
```

### 9. MIDDLEWARE (3 files)
```
app/Http/Middleware/
├── EnsureTenantAccess.php
├── CheckSubscription.php
├── TrackLastActivity.php
```

### 10. SEEDERS (2 files)
```
database/seeders/
├── TenantSeeder.php
├── RolePermissionSeeder.php
```

### 11. TESTS (5 files)
```
tests/Feature/
├── AuthTest.php
├── TenantTest.php
├── UserTest.php
├── TeamTest.php
├── InvitationTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### AuthController.php
```php
<?php
namespace App\Http\Controllers\Api;

class AuthController extends ApiController
{
    // POST /api/auth/register - Register new tenant + admin user
    public function register(RegisterRequest $request);
    
    // POST /api/auth/login - Login with email/password
    public function login(LoginRequest $request);
    
    // POST /api/auth/logout - Logout (revoke token)
    public function logout();
    
    // POST /api/auth/refresh - Refresh token
    public function refresh();
    
    // GET /api/auth/me - Get current user
    public function me();
    
    // POST /api/auth/forgot-password - Send reset link
    public function forgotPassword(ForgotPasswordRequest $request);
    
    // POST /api/auth/reset-password - Reset password
    public function resetPassword(ResetPasswordRequest $request);
    
    // POST /api/auth/verify-email/{id}/{hash} - Verify email
    public function verifyEmail($id, $hash);
    
    // POST /api/auth/two-factor/enable - Enable 2FA
    public function enableTwoFactor();
    
    // POST /api/auth/two-factor/verify - Verify 2FA code
    public function verifyTwoFactor(Request $request);
}
```

### TenantController.php
```php
<?php
namespace App\Http\Controllers\Api;

class TenantController extends ApiController
{
    // GET /api/tenant - Get current tenant
    public function show();
    
    // PUT /api/tenant - Update tenant settings
    public function update(UpdateTenantRequest $request);
    
    // PUT /api/tenant/branding - Update branding
    public function updateBranding(Request $request);
    
    // GET /api/tenant/billing - Get billing info
    public function billing();
    
    // POST /api/tenant/subscription - Update subscription
    public function updateSubscription(Request $request);
    
    // GET /api/tenant/usage - Get usage stats
    public function usage();
    
    // DELETE /api/tenant - Delete tenant (with confirmation)
    public function destroy(Request $request);
}
```

### UserController.php
```php
<?php
namespace App\Http\Controllers\Api;

class UserController extends ApiController
{
    // GET /api/users - List users
    public function index(Request $request);
    
    // POST /api/users - Create user
    public function store(StoreUserRequest $request);
    
    // GET /api/users/{user} - Get user
    public function show(User $user);
    
    // PUT /api/users/{user} - Update user
    public function update(UpdateUserRequest $request, User $user);
    
    // DELETE /api/users/{user} - Delete user
    public function destroy(User $user);
    
    // POST /api/users/{user}/deactivate - Deactivate user
    public function deactivate(User $user);
    
    // POST /api/users/{user}/activate - Reactivate user
    public function activate(User $user);
    
    // PUT /api/users/{user}/roles - Update user roles
    public function updateRoles(Request $request, User $user);
    
    // GET /api/users/{user}/activity - Get user activity log
    public function activity(User $user);
}
```

### TeamController.php
```php
<?php
namespace App\Http\Controllers\Api;

class TeamController extends ApiController
{
    // GET /api/teams - List teams
    public function index();
    
    // POST /api/teams - Create team
    public function store(StoreTeamRequest $request);
    
    // GET /api/teams/{team} - Get team
    public function show(Team $team);
    
    // PUT /api/teams/{team} - Update team
    public function update(Request $request, Team $team);
    
    // DELETE /api/teams/{team} - Delete team
    public function destroy(Team $team);
    
    // GET /api/teams/{team}/members - Get team members
    public function members(Team $team);
    
    // POST /api/teams/{team}/members - Add member
    public function addMember(Request $request, Team $team);
    
    // DELETE /api/teams/{team}/members/{user} - Remove member
    public function removeMember(Team $team, User $user);
    
    // PUT /api/teams/{team}/lead - Set team lead
    public function setLead(Request $request, Team $team);
}
```

---

## 🗄️ DATABASE TABLES (Already exist in migrations)
- tenants
- users  
- teams
- team_members
- user_availability
- invitations
- password_reset_tokens
- sessions

---

## 🔗 ROUTES TO DEFINE

```php
// routes/api/auth.php

Route::prefix('auth')->group(function () {
    Route::post('register', [AuthController::class, 'register']);
    Route::post('login', [AuthController::class, 'login']);
    Route::post('forgot-password', [AuthController::class, 'forgotPassword']);
    Route::post('reset-password', [AuthController::class, 'resetPassword']);
    
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('logout', [AuthController::class, 'logout']);
        Route::post('refresh', [AuthController::class, 'refresh']);
        Route::get('me', [AuthController::class, 'me']);
        Route::post('two-factor/enable', [AuthController::class, 'enableTwoFactor']);
        Route::post('two-factor/verify', [AuthController::class, 'verifyTwoFactor']);
    });
});

Route::middleware('auth:sanctum')->group(function () {
    // Tenant
    Route::get('tenant', [TenantController::class, 'show']);
    Route::put('tenant', [TenantController::class, 'update']);
    Route::put('tenant/branding', [TenantController::class, 'updateBranding']);
    Route::get('tenant/billing', [TenantController::class, 'billing']);
    Route::get('tenant/usage', [TenantController::class, 'usage']);
    
    // Users
    Route::apiResource('users', UserController::class);
    Route::post('users/{user}/deactivate', [UserController::class, 'deactivate']);
    Route::post('users/{user}/activate', [UserController::class, 'activate']);
    Route::put('users/{user}/roles', [UserController::class, 'updateRoles']);
    
    // Teams
    Route::apiResource('teams', TeamController::class);
    Route::get('teams/{team}/members', [TeamController::class, 'members']);
    Route::post('teams/{team}/members', [TeamController::class, 'addMember']);
    Route::delete('teams/{team}/members/{user}', [TeamController::class, 'removeMember']);
    
    // Roles
    Route::apiResource('roles', RoleController::class);
    
    // Profile
    Route::get('profile', [ProfileController::class, 'show']);
    Route::put('profile', [ProfileController::class, 'update']);
    Route::put('profile/password', [ProfileController::class, 'updatePassword']);
    Route::put('profile/avatar', [ProfileController::class, 'updateAvatar']);
    Route::get('profile/notifications', [ProfileController::class, 'notifications']);
    
    // Invitations
    Route::post('invitations', [InvitationController::class, 'store']);
    Route::delete('invitations/{invitation}', [InvitationController::class, 'destroy']);
    Route::post('invitations/{invitation}/resend', [InvitationController::class, 'resend']);
});

// Public invitation acceptance
Route::get('invitations/{token}/accept', [InvitationController::class, 'accept']);
Route::post('invitations/{token}/complete', [InvitationController::class, 'complete']);
```

---

## ✅ COMPLETION CHECKLIST

- [ ] All models created with relationships
- [ ] All controllers with full CRUD
- [ ] All services with business logic
- [ ] All form requests with validation
- [ ] All routes defined
- [ ] All policies created
- [ ] All middleware created
- [ ] All jobs created
- [ ] All notifications created
- [ ] All seeders created
- [ ] All tests passing

---

## 📤 OUTPUT LOCATION

Save all files to: `/home/claude/business-platform/`

When complete, update: `/home/claude/business-platform/docs/PROGRESS_BOT_1.md`
