# 🤖 BOT 4: ACCOUNTING - BILLS & BANKING MODULE

## 📋 YOUR RESPONSIBILITY
Build vendor management, bills/expenses, bill payments, bank accounts, transactions, and reconciliation.

---

## 📁 FILES TO CREATE

### 1. MODELS (12 files)
```
app/Models/
├── Vendor.php ✅ EXISTS - ENHANCE
├── Bill.php - CREATE
├── BillItem.php - CREATE
├── BillPayment.php - CREATE
├── Expense.php - CREATE
├── ExpenseCategory.php - CREATE
├── BankAccount.php - CREATE
├── BankTransaction.php - CREATE
├── BankReconciliation.php - CREATE
├── BankRule.php - CREATE
├── ChartOfAccount.php ✅ EXISTS - ENHANCE
├── JournalEntry.php ✅ EXISTS - ENHANCE
```

### 2. CONTROLLERS (8 files)
```
app/Http/Controllers/Api/
├── VendorController.php - CREATE
├── BillController.php - CREATE
├── BillPaymentController.php - CREATE
├── ExpenseController.php - CREATE
├── BankAccountController.php - CREATE
├── BankTransactionController.php - CREATE
├── BankReconciliationController.php - CREATE
├── ChartOfAccountController.php - CREATE
```

### 3. SERVICES (6 files)
```
app/Services/
├── VendorService.php - CREATE
├── BillService.php - CREATE
├── ExpenseService.php - CREATE
├── BankingService.php - CREATE
├── ReconciliationService.php - CREATE
├── PlaidService.php - CREATE (bank sync)
```

### 4. FORM REQUESTS (12 files)
```
app/Http/Requests/
├── Vendor/StoreVendorRequest.php
├── Vendor/UpdateVendorRequest.php
├── Bill/StoreBillRequest.php
├── Bill/UpdateBillRequest.php
├── BillPayment/StoreBillPaymentRequest.php
├── Expense/StoreExpenseRequest.php
├── Expense/UpdateExpenseRequest.php
├── BankAccount/StoreBankAccountRequest.php
├── BankTransaction/StoreBankTransactionRequest.php
├── BankTransaction/MatchTransactionRequest.php
├── BankRule/StoreBankRuleRequest.php
├── ChartOfAccount/StoreAccountRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/accounting.php - CREATE
```

### 6. JOBS (5 files)
```
app/Jobs/
├── SyncBankTransactions.php
├── AutoCategorizeTransactions.php
├── SendBillReminder.php
├── MarkOverdueBills.php
├── GenerateAccountingReports.php
```

### 7. EVENTS & LISTENERS (6 files)
```
app/Events/
├── BillCreated.php
├── BillPaid.php
├── ExpenseApproved.php
├── BankTransactionImported.php

app/Listeners/
├── CreateJournalEntryOnBill.php
├── CreateJournalEntryOnBillPayment.php
├── CreateJournalEntryOnExpense.php
├── UpdateVendorBalance.php
```

### 8. NOTIFICATIONS (3 files)
```
app/Notifications/
├── BillDueReminderNotification.php
├── BillOverdueNotification.php
├── ExpenseApprovedNotification.php
```

### 9. SEEDERS (2 files)
```
database/seeders/
├── ChartOfAccountSeeder.php
├── VendorSeeder.php
```

### 10. TESTS (6 files)
```
tests/Feature/
├── VendorTest.php
├── BillTest.php
├── ExpenseTest.php
├── BankAccountTest.php
├── BankReconciliationTest.php
├── ChartOfAccountTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### VendorController.php
```php
<?php
namespace App\Http\Controllers\Api;

class VendorController extends ApiController
{
    public function index(Request $request);      // List vendors
    public function store(StoreVendorRequest $request);  // Create
    public function show(Vendor $vendor);         // Show with bills
    public function update(UpdateVendorRequest $request, Vendor $vendor);
    public function destroy(Vendor $vendor);      // Soft delete
    public function bills(Vendor $vendor);        // Vendor bills
    public function payments(Vendor $vendor);     // Payment history
    public function statement(Vendor $vendor);    // Account statement
}
```

### BillController.php
```php
<?php
namespace App\Http\Controllers\Api;

class BillController extends ApiController
{
    public function index(Request $request);      // List with filters
    public function store(StoreBillRequest $request);   // Create bill
    public function show(Bill $bill);             // Show bill
    public function update(UpdateBillRequest $request, Bill $bill);
    public function destroy(Bill $bill);          // Delete draft only
    public function recordPayment(StoreBillPaymentRequest $request, Bill $bill);
    public function markReceived(Bill $bill);     // Mark as received
    public function void(Bill $bill);             // Void bill
    public function duplicate(Bill $bill);        // Duplicate
    public function summary(Request $request);    // Bills summary
    public function aging();                      // Aging report (payables)
    public function overdue();                    // Overdue bills
}
```

### ExpenseController.php
```php
<?php
namespace App\Http\Controllers\Api;

class ExpenseController extends ApiController
{
    public function index(Request $request);      // List expenses
    public function store(StoreExpenseRequest $request);  // Create
    public function show(Expense $expense);       // Show
    public function update(UpdateExpenseRequest $request, Expense $expense);
    public function destroy(Expense $expense);    // Delete pending only
    public function approve(Expense $expense);    // Approve expense
    public function reject(Request $request, Expense $expense);  // Reject
    public function uploadReceipt(Request $request, Expense $expense);
    public function byCategory(Request $request); // Group by category
    public function summary(Request $request);    // Expense summary
}
```

### BankAccountController.php
```php
<?php
namespace App\Http\Controllers\Api;

class BankAccountController extends ApiController
{
    public function index();                      // List bank accounts
    public function store(StoreBankAccountRequest $request);  // Add account
    public function show(BankAccount $account);   // Show with balance
    public function update(Request $request, BankAccount $account);
    public function destroy(BankAccount $account);
    public function transactions(BankAccount $account, Request $request);
    public function syncPlaid(BankAccount $account);  // Sync with Plaid
    public function connectPlaid(Request $request);   // Connect new Plaid
    public function disconnectPlaid(BankAccount $account);
    public function importTransactions(Request $request, BankAccount $account); // CSV import
}
```

### BankReconciliationController.php
```php
<?php
namespace App\Http\Controllers\Api;

class BankReconciliationController extends ApiController
{
    public function index(BankAccount $account);  // List reconciliations
    public function start(Request $request, BankAccount $account);  // Start new
    public function show(BankReconciliation $reconciliation);
    public function matchTransaction(Request $request, BankReconciliation $recon);
    public function unmatchTransaction(BankTransaction $transaction);
    public function excludeTransaction(BankTransaction $transaction);
    public function complete(BankReconciliation $reconciliation);
    public function cancel(BankReconciliation $reconciliation);
}
```

### ChartOfAccountController.php
```php
<?php
namespace App\Http\Controllers\Api;

class ChartOfAccountController extends ApiController
{
    public function index(Request $request);      // List accounts (tree)
    public function store(StoreAccountRequest $request);  // Create account
    public function show(ChartOfAccount $account);  // Show with balance
    public function update(Request $request, ChartOfAccount $account);
    public function destroy(ChartOfAccount $account);  // Only if no transactions
    public function ledger(ChartOfAccount $account, Request $request);  // Account ledger
    public function trialBalance(Request $request);  // Trial balance
    public function balanceSheet(Request $request);  // Balance sheet
    public function incomeStatement(Request $request);  // P&L
}
```

---

## 🗄️ DATABASE TABLES
- vendors
- bills
- bill_items
- bill_payments
- expenses
- bank_accounts
- bank_transactions
- bank_reconciliations
- bank_rules
- chart_of_accounts
- journal_entries
- journal_entry_lines

---

## 🔗 ROUTES TO DEFINE

```php
// routes/api/accounting.php

Route::middleware('auth:sanctum')->group(function () {
    // Vendors
    Route::apiResource('vendors', VendorController::class);
    Route::get('vendors/{vendor}/bills', [VendorController::class, 'bills']);
    Route::get('vendors/{vendor}/payments', [VendorController::class, 'payments']);
    Route::get('vendors/{vendor}/statement', [VendorController::class, 'statement']);
    
    // Bills
    Route::apiResource('bills', BillController::class);
    Route::post('bills/{bill}/payment', [BillController::class, 'recordPayment']);
    Route::post('bills/{bill}/received', [BillController::class, 'markReceived']);
    Route::post('bills/{bill}/void', [BillController::class, 'void']);
    Route::post('bills/{bill}/duplicate', [BillController::class, 'duplicate']);
    Route::get('bills-summary', [BillController::class, 'summary']);
    Route::get('bills-aging', [BillController::class, 'aging']);
    Route::get('bills-overdue', [BillController::class, 'overdue']);
    
    // Bill Payments
    Route::apiResource('bill-payments', BillPaymentController::class);
    
    // Expenses
    Route::apiResource('expenses', ExpenseController::class);
    Route::post('expenses/{expense}/approve', [ExpenseController::class, 'approve']);
    Route::post('expenses/{expense}/reject', [ExpenseController::class, 'reject']);
    Route::post('expenses/{expense}/receipt', [ExpenseController::class, 'uploadReceipt']);
    Route::get('expenses-by-category', [ExpenseController::class, 'byCategory']);
    Route::get('expenses-summary', [ExpenseController::class, 'summary']);
    
    // Bank Accounts
    Route::apiResource('bank-accounts', BankAccountController::class);
    Route::get('bank-accounts/{account}/transactions', [BankAccountController::class, 'transactions']);
    Route::post('bank-accounts/{account}/sync', [BankAccountController::class, 'syncPlaid']);
    Route::post('bank-accounts/connect-plaid', [BankAccountController::class, 'connectPlaid']);
    Route::delete('bank-accounts/{account}/plaid', [BankAccountController::class, 'disconnectPlaid']);
    Route::post('bank-accounts/{account}/import', [BankAccountController::class, 'importTransactions']);
    
    // Bank Transactions
    Route::apiResource('bank-transactions', BankTransactionController::class);
    Route::post('bank-transactions/{transaction}/categorize', [BankTransactionController::class, 'categorize']);
    Route::post('bank-transactions/{transaction}/match', [BankTransactionController::class, 'match']);
    Route::post('bank-transactions/{transaction}/unmatch', [BankTransactionController::class, 'unmatch']);
    Route::post('bank-transactions/{transaction}/exclude', [BankTransactionController::class, 'exclude']);
    
    // Bank Rules
    Route::apiResource('bank-rules', BankRuleController::class);
    
    // Reconciliation
    Route::get('bank-accounts/{account}/reconciliations', [BankReconciliationController::class, 'index']);
    Route::post('bank-accounts/{account}/reconciliations', [BankReconciliationController::class, 'start']);
    Route::get('reconciliations/{reconciliation}', [BankReconciliationController::class, 'show']);
    Route::post('reconciliations/{reconciliation}/match', [BankReconciliationController::class, 'matchTransaction']);
    Route::post('reconciliations/{reconciliation}/complete', [BankReconciliationController::class, 'complete']);
    Route::post('reconciliations/{reconciliation}/cancel', [BankReconciliationController::class, 'cancel']);
    
    // Chart of Accounts
    Route::apiResource('accounts', ChartOfAccountController::class);
    Route::get('accounts/{account}/ledger', [ChartOfAccountController::class, 'ledger']);
    
    // Reports
    Route::get('reports/trial-balance', [ChartOfAccountController::class, 'trialBalance']);
    Route::get('reports/balance-sheet', [ChartOfAccountController::class, 'balanceSheet']);
    Route::get('reports/income-statement', [ChartOfAccountController::class, 'incomeStatement']);
    Route::get('reports/cash-flow', [ReportController::class, 'cashFlow']);
});
```

---

## 🏦 CHART OF ACCOUNTS STRUCTURE

```
1000 - Assets
├── 1100 - Current Assets
│   ├── 1110 - Cash
│   ├── 1120 - Bank Accounts
│   ├── 1130 - Accounts Receivable
│   └── 1140 - Inventory
├── 1200 - Fixed Assets
│   ├── 1210 - Equipment
│   └── 1220 - Accumulated Depreciation

2000 - Liabilities
├── 2100 - Current Liabilities
│   ├── 2110 - Accounts Payable
│   ├── 2120 - VAT Payable
│   └── 2130 - Accrued Expenses
├── 2200 - Long-term Liabilities
│   └── 2210 - Loans Payable

3000 - Equity
├── 3100 - Owner's Capital
└── 3200 - Retained Earnings

4000 - Revenue
├── 4100 - Sales Revenue
├── 4200 - Service Revenue
└── 4300 - Other Income

5000 - Expenses
├── 5100 - Cost of Goods Sold
├── 5200 - Operating Expenses
│   ├── 5210 - Rent
│   ├── 5220 - Utilities
│   ├── 5230 - Salaries
│   └── 5240 - Marketing
└── 5300 - Other Expenses
```

---

## ✅ COMPLETION CHECKLIST

- [ ] All models with relationships
- [ ] Vendor CRUD + statement
- [ ] Bill CRUD + payments
- [ ] Expense CRUD + approval workflow
- [ ] Bank accounts + Plaid integration
- [ ] Transaction import/categorization
- [ ] Bank reconciliation
- [ ] Chart of accounts
- [ ] Journal entries (auto-generated)
- [ ] Financial reports (trial balance, P&L, balance sheet)
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT LOCATION

Save all files to: `/home/claude/business-platform/`

When complete, update: `/home/claude/business-platform/docs/PROGRESS_BOT_4.md`
