# 🤖 BOT 5: TASK MANAGEMENT MODULE

## 📋 YOUR RESPONSIBILITY
Build project management, tasks, subtasks, time tracking, timesheets, milestones, and goals.

---

## 📁 FILES TO CREATE

### 1. MODELS (18 files)
```
app/Models/
├── Workspace.php - CREATE
├── Project.php ✅ EXISTS - ENHANCE
├── ProjectMember.php - CREATE
├── TaskStatus.php ✅ EXISTS - ENHANCE
├── TaskGroup.php - CREATE
├── Task.php ✅ EXISTS - ENHANCE
├── TaskAssignee.php - CREATE
├── TaskDependency.php - CREATE
├── TaskChecklist.php - CREATE
├── ChecklistItem.php - CREATE
├── TaskComment.php - CREATE
├── TaskAttachment.php - CREATE
├── TaskActivity.php - CREATE
├── TimeEntry.php ✅ EXISTS - ENHANCE
├── Timesheet.php - CREATE
├── Milestone.php - CREATE
├── Goal.php - CREATE
├── TaskAutomation.php - CREATE
```

### 2. CONTROLLERS (10 files)
```
app/Http/Controllers/Api/
├── WorkspaceController.php - CREATE
├── ProjectController.php ✅ EXISTS - ENHANCE
├── TaskController.php ✅ EXISTS - ENHANCE
├── TaskCommentController.php - CREATE
├── TaskChecklistController.php - CREATE
├── TimeEntryController.php - CREATE
├── TimesheetController.php - CREATE
├── MilestoneController.php - CREATE
├── GoalController.php - CREATE
├── TaskAutomationController.php - CREATE
```

### 3. SERVICES (6 files)
```
app/Services/
├── ProjectService.php - CREATE
├── TaskService.php - CREATE
├── TimeTrackingService.php - CREATE
├── TimesheetService.php - CREATE
├── GanttService.php - CREATE
├── TaskAutomationService.php - CREATE
```

### 4. FORM REQUESTS (14 files)
```
app/Http/Requests/
├── Workspace/StoreWorkspaceRequest.php
├── Project/StoreProjectRequest.php
├── Project/UpdateProjectRequest.php
├── Task/StoreTaskRequest.php
├── Task/UpdateTaskRequest.php
├── Task/MoveTaskRequest.php
├── Task/BulkUpdateTasksRequest.php
├── TaskComment/StoreCommentRequest.php
├── TimeEntry/StoreTimeEntryRequest.php
├── TimeEntry/UpdateTimeEntryRequest.php
├── Timesheet/SubmitTimesheetRequest.php
├── Milestone/StoreMilestoneRequest.php
├── Goal/StoreGoalRequest.php
├── TaskAutomation/StoreAutomationRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/tasks.php - CREATE
```

### 6. JOBS (5 files)
```
app/Jobs/
├── RecalculateProjectProgress.php
├── SendTaskDueReminder.php
├── ProcessTaskAutomation.php
├── GenerateTimesheetReport.php
├── CreateRecurringTasks.php
```

### 7. EVENTS & LISTENERS (8 files)
```
app/Events/
├── TaskCreated.php
├── TaskUpdated.php
├── TaskCompleted.php
├── TaskCommentAdded.php
├── TimeEntryLogged.php

app/Listeners/
├── LogTaskActivity.php
├── NotifyTaskAssignee.php
├── UpdateProjectProgress.php
├── RunTaskAutomations.php
```

### 8. NOTIFICATIONS (4 files)
```
app/Notifications/
├── TaskAssignedNotification.php
├── TaskDueReminderNotification.php
├── TaskCommentNotification.php
├── TimesheetApprovedNotification.php
```

### 9. POLICIES (4 files)
```
app/Policies/
├── ProjectPolicy.php
├── TaskPolicy.php
├── TimeEntryPolicy.php
├── TimesheetPolicy.php
```

### 10. SEEDERS (2 files)
```
database/seeders/
├── ProjectSeeder.php
├── TaskSeeder.php
```

### 11. TESTS (7 files)
```
tests/Feature/
├── WorkspaceTest.php
├── ProjectTest.php
├── TaskTest.php
├── TaskCommentTest.php
├── TimeEntryTest.php
├── TimesheetTest.php
├── TaskAutomationTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### ProjectController.php (ENHANCE)
```php
<?php
namespace App\Http\Controllers\Api;

class ProjectController extends ApiController
{
    public function index(Request $request);      // List projects
    public function store(StoreProjectRequest $request);  // Create
    public function show(Project $project);       // Show with tasks
    public function update(UpdateProjectRequest $request, Project $project);
    public function destroy(Project $project);    // Archive/delete
    public function archive(Project $project);    // Archive
    public function unarchive(Project $project);  // Unarchive
    
    // Members
    public function members(Project $project);
    public function addMember(Request $request, Project $project);
    public function updateMember(Request $request, Project $project, User $user);
    public function removeMember(Project $project, User $user);
    
    // Views
    public function kanban(Project $project);     // Kanban board
    public function gantt(Project $project);      // Gantt chart data
    public function calendar(Project $project);   // Calendar view
    public function list(Project $project);       // List view
    
    // Stats
    public function stats(Project $project);      // Project statistics
    public function burndown(Project $project);   // Burndown chart
    public function velocity(Project $project);   // Velocity chart
    
    // Templates
    public function saveAsTemplate(Project $project);
    public function createFromTemplate(Request $request);
}
```

### TaskController.php (ENHANCE)
```php
<?php
namespace App\Http\Controllers\Api;

class TaskController extends ApiController
{
    public function index(Request $request);      // List with filters
    public function store(StoreTaskRequest $request);  // Create
    public function show(Task $task);             // Show with details
    public function update(UpdateTaskRequest $request, Task $task);
    public function destroy(Task $task);          // Delete
    
    // Actions
    public function move(MoveTaskRequest $request, Task $task);  // Move status/position
    public function complete(Task $task);         // Mark complete
    public function reopen(Task $task);           // Reopen
    public function duplicate(Task $task);        // Duplicate task
    
    // Bulk
    public function bulkUpdate(BulkUpdateTasksRequest $request);
    public function bulkDelete(Request $request);
    public function bulkMove(Request $request);
    
    // Subtasks
    public function subtasks(Task $task);
    public function addSubtask(StoreTaskRequest $request, Task $task);
    
    // Dependencies
    public function dependencies(Task $task);
    public function addDependency(Request $request, Task $task);
    public function removeDependency(Task $task, Task $dependency);
    
    // Assignees
    public function assignees(Task $task);
    public function assign(Request $request, Task $task);
    public function unassign(Task $task, User $user);
    
    // Time
    public function timeEntries(Task $task);
    public function logTime(StoreTimeEntryRequest $request, Task $task);
    public function startTimer(Task $task);
    public function stopTimer(Task $task);
    
    // Comments & Attachments
    public function comments(Task $task);
    public function comment(StoreCommentRequest $request, Task $task);
    public function attachments(Task $task);
    public function attach(Request $request, Task $task);
    
    // Activity
    public function activity(Task $task);
    
    // My Tasks
    public function myTasks(Request $request);
    public function myTasksToday();
    public function myTasksOverdue();
}
```

### TimeEntryController.php
```php
<?php
namespace App\Http\Controllers\Api;

class TimeEntryController extends ApiController
{
    public function index(Request $request);      // List time entries
    public function store(StoreTimeEntryRequest $request);  // Log time
    public function show(TimeEntry $entry);       // Show entry
    public function update(UpdateTimeEntryRequest $request, TimeEntry $entry);
    public function destroy(TimeEntry $entry);    // Delete
    
    public function startTimer(Request $request); // Start timer
    public function stopTimer(TimeEntry $entry);  // Stop timer
    public function getRunningTimer();            // Get current timer
    
    public function myTime(Request $request);     // My time entries
    public function byProject(Request $request);  // Group by project
    public function byUser(Request $request);     // Group by user
    public function summary(Request $request);    // Time summary/report
}
```

### TimesheetController.php
```php
<?php
namespace App\Http\Controllers\Api;

class TimesheetController extends ApiController
{
    public function index(Request $request);      // List timesheets
    public function show(Timesheet $timesheet);   // Show timesheet
    public function current();                    // Current week timesheet
    public function week(Request $request);       // Specific week
    
    public function submit(Timesheet $timesheet); // Submit for approval
    public function approve(Timesheet $timesheet);// Approve
    public function reject(Request $request, Timesheet $timesheet);
    
    public function pending();                    // Pending approvals
    public function report(Request $request);     // Timesheet report
}
```

---

## 🗄️ DATABASE TABLES
- workspaces
- projects
- project_members
- task_statuses
- task_groups
- tasks
- task_assignees
- task_dependencies
- task_tag
- task_checklists
- checklist_items
- task_comments
- task_attachments
- task_activities
- time_entries
- timesheets
- milestones
- goals
- goal_project
- task_automations

---

## 🔗 ROUTES TO DEFINE

```php
// routes/api/tasks.php

Route::middleware('auth:sanctum')->group(function () {
    // Workspaces
    Route::apiResource('workspaces', WorkspaceController::class);
    
    // Projects
    Route::apiResource('projects', ProjectController::class);
    Route::post('projects/{project}/archive', [ProjectController::class, 'archive']);
    Route::post('projects/{project}/unarchive', [ProjectController::class, 'unarchive']);
    Route::get('projects/{project}/members', [ProjectController::class, 'members']);
    Route::post('projects/{project}/members', [ProjectController::class, 'addMember']);
    Route::put('projects/{project}/members/{user}', [ProjectController::class, 'updateMember']);
    Route::delete('projects/{project}/members/{user}', [ProjectController::class, 'removeMember']);
    Route::get('projects/{project}/kanban', [ProjectController::class, 'kanban']);
    Route::get('projects/{project}/gantt', [ProjectController::class, 'gantt']);
    Route::get('projects/{project}/calendar', [ProjectController::class, 'calendar']);
    Route::get('projects/{project}/stats', [ProjectController::class, 'stats']);
    Route::post('projects/{project}/save-template', [ProjectController::class, 'saveAsTemplate']);
    Route::post('projects/from-template', [ProjectController::class, 'createFromTemplate']);
    
    // Task Statuses
    Route::apiResource('projects.statuses', TaskStatusController::class)->shallow();
    Route::post('projects/{project}/statuses/reorder', [TaskStatusController::class, 'reorder']);
    
    // Task Groups
    Route::apiResource('projects.groups', TaskGroupController::class)->shallow();
    
    // Tasks
    Route::apiResource('tasks', TaskController::class);
    Route::post('tasks/{task}/move', [TaskController::class, 'move']);
    Route::post('tasks/{task}/complete', [TaskController::class, 'complete']);
    Route::post('tasks/{task}/reopen', [TaskController::class, 'reopen']);
    Route::post('tasks/{task}/duplicate', [TaskController::class, 'duplicate']);
    Route::post('tasks/bulk-update', [TaskController::class, 'bulkUpdate']);
    Route::post('tasks/bulk-delete', [TaskController::class, 'bulkDelete']);
    
    // Subtasks
    Route::get('tasks/{task}/subtasks', [TaskController::class, 'subtasks']);
    Route::post('tasks/{task}/subtasks', [TaskController::class, 'addSubtask']);
    
    // Dependencies
    Route::get('tasks/{task}/dependencies', [TaskController::class, 'dependencies']);
    Route::post('tasks/{task}/dependencies', [TaskController::class, 'addDependency']);
    Route::delete('tasks/{task}/dependencies/{dependency}', [TaskController::class, 'removeDependency']);
    
    // Assignees
    Route::post('tasks/{task}/assign', [TaskController::class, 'assign']);
    Route::delete('tasks/{task}/assignees/{user}', [TaskController::class, 'unassign']);
    
    // Comments
    Route::get('tasks/{task}/comments', [TaskController::class, 'comments']);
    Route::post('tasks/{task}/comments', [TaskController::class, 'comment']);
    Route::put('comments/{comment}', [TaskCommentController::class, 'update']);
    Route::delete('comments/{comment}', [TaskCommentController::class, 'destroy']);
    
    // Checklists
    Route::apiResource('tasks.checklists', TaskChecklistController::class)->shallow();
    Route::post('checklists/{checklist}/items', [TaskChecklistController::class, 'addItem']);
    Route::put('checklist-items/{item}', [TaskChecklistController::class, 'updateItem']);
    Route::post('checklist-items/{item}/toggle', [TaskChecklistController::class, 'toggleItem']);
    Route::delete('checklist-items/{item}', [TaskChecklistController::class, 'deleteItem']);
    
    // Attachments
    Route::get('tasks/{task}/attachments', [TaskController::class, 'attachments']);
    Route::post('tasks/{task}/attachments', [TaskController::class, 'attach']);
    Route::delete('attachments/{attachment}', [TaskAttachmentController::class, 'destroy']);
    
    // Time Tracking
    Route::get('tasks/{task}/time-entries', [TaskController::class, 'timeEntries']);
    Route::post('tasks/{task}/time', [TaskController::class, 'logTime']);
    Route::post('tasks/{task}/timer/start', [TaskController::class, 'startTimer']);
    Route::post('tasks/{task}/timer/stop', [TaskController::class, 'stopTimer']);
    
    // Time Entries
    Route::apiResource('time-entries', TimeEntryController::class);
    Route::post('time-entries/start', [TimeEntryController::class, 'startTimer']);
    Route::post('time-entries/{entry}/stop', [TimeEntryController::class, 'stopTimer']);
    Route::get('time-entries/running', [TimeEntryController::class, 'getRunningTimer']);
    Route::get('time-entries-summary', [TimeEntryController::class, 'summary']);
    
    // Timesheets
    Route::get('timesheets', [TimesheetController::class, 'index']);
    Route::get('timesheets/current', [TimesheetController::class, 'current']);
    Route::get('timesheets/week', [TimesheetController::class, 'week']);
    Route::get('timesheets/{timesheet}', [TimesheetController::class, 'show']);
    Route::post('timesheets/{timesheet}/submit', [TimesheetController::class, 'submit']);
    Route::post('timesheets/{timesheet}/approve', [TimesheetController::class, 'approve']);
    Route::post('timesheets/{timesheet}/reject', [TimesheetController::class, 'reject']);
    Route::get('timesheets-pending', [TimesheetController::class, 'pending']);
    
    // Milestones
    Route::apiResource('projects.milestones', MilestoneController::class)->shallow();
    
    // Goals
    Route::apiResource('goals', GoalController::class);
    Route::post('goals/{goal}/update-progress', [GoalController::class, 'updateProgress']);
    
    // My Tasks
    Route::get('my/tasks', [TaskController::class, 'myTasks']);
    Route::get('my/tasks/today', [TaskController::class, 'myTasksToday']);
    Route::get('my/tasks/overdue', [TaskController::class, 'myTasksOverdue']);
    Route::get('my/time', [TimeEntryController::class, 'myTime']);
    
    // Automations
    Route::apiResource('task-automations', TaskAutomationController::class);
});
```

---

## ✅ COMPLETION CHECKLIST

- [ ] All models with relationships
- [ ] Project CRUD + members + views
- [ ] Task CRUD + subtasks + dependencies
- [ ] Kanban board data
- [ ] Gantt chart data
- [ ] Time tracking with timer
- [ ] Timesheets + approval
- [ ] Comments + mentions
- [ ] Checklists
- [ ] File attachments
- [ ] Task automations
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT LOCATION

Save all files to: `/home/claude/business-platform/`

When complete, update: `/home/claude/business-platform/docs/PROGRESS_BOT_5.md`
