import { defineStore } from 'pinia';
import api from '../api';

export const useEmployeeStore = defineStore('employees', {
    state: () => ({
        employees: [],
        employee: null,
        departments: [],
        loading: false,
        error: null,
    }),

    getters: {
        activeEmployees: (state) => state.employees.filter(e => e.status === 'active'),
        byDepartment: (state) => (deptId) => state.employees.filter(e => e.department_id === deptId),
    },

    actions: {
        async fetchEmployees(params = {}) {
            this.loading = true;
            try {
                const { data } = await api.get('/employees', { params });
                this.employees = data.data;
            } catch (error) {
                this.error = error.message;
            } finally {
                this.loading = false;
            }
        },

        async fetchEmployee(id) {
            this.loading = true;
            try {
                const { data } = await api.get(`/employees/${id}`);
                this.employee = data.data;
                return this.employee;
            } catch (error) {
                this.error = error.message;
            } finally {
                this.loading = false;
            }
        },

        async fetchDepartments() {
            const { data } = await api.get('/departments');
            this.departments = data.data;
            return this.departments;
        },

        async createEmployee(employeeData) {
            const { data } = await api.post('/employees', employeeData);
            this.employees.unshift(data.data);
            return data.data;
        },

        async updateEmployee(id, employeeData) {
            const { data } = await api.put(`/employees/${id}`, employeeData);
            const index = this.employees.findIndex(e => e.id === id);
            if (index !== -1) this.employees[index] = data.data;
            return data.data;
        },

        async updateSalary(employeeId, salaryData) {
            const { data } = await api.post(`/employees/${employeeId}/salary`, salaryData);
            return data.data;
        },

        async uploadDocument(employeeId, formData) {
            const { data } = await api.post(`/employees/${employeeId}/documents`, formData, {
                headers: { 'Content-Type': 'multipart/form-data' },
            });
            return data.data;
        },
    },
});
