import { defineStore } from 'pinia';
import api from '../api';

export const useOpportunityStore = defineStore('opportunities', {
    state: () => ({
        opportunities: [],
        opportunity: null,
        pipelines: [],
        loading: false,
        error: null,
    }),

    getters: {
        openOpportunities: (state) => state.opportunities.filter(o => o.status === 'open'),
        totalValue: (state) => state.opportunities.reduce((sum, o) => sum + parseFloat(o.value || 0), 0),
        weightedValue: (state) => state.opportunities.reduce((sum, o) => sum + (parseFloat(o.value || 0) * (o.probability / 100)), 0),
        byStage: (state) => (stageId) => state.opportunities.filter(o => o.stage_id === stageId),
    },

    actions: {
        async fetchOpportunities(params = {}) {
            this.loading = true;
            try {
                const { data } = await api.get('/opportunities', { params });
                this.opportunities = data.data;
            } catch (error) {
                this.error = error.message;
            } finally {
                this.loading = false;
            }
        },

        async fetchOpportunity(id) {
            this.loading = true;
            try {
                const { data } = await api.get(`/opportunities/${id}`);
                this.opportunity = data.data;
                return this.opportunity;
            } catch (error) {
                this.error = error.message;
            } finally {
                this.loading = false;
            }
        },

        async fetchPipelines() {
            const { data } = await api.get('/pipelines');
            this.pipelines = data.data;
            return this.pipelines;
        },

        async createOpportunity(oppData) {
            const { data } = await api.post('/opportunities', oppData);
            this.opportunities.unshift(data.data);
            return data.data;
        },

        async updateOpportunity(id, oppData) {
            const { data } = await api.put(`/opportunities/${id}`, oppData);
            const index = this.opportunities.findIndex(o => o.id === id);
            if (index !== -1) this.opportunities[index] = data.data;
            return data.data;
        },

        async updateStage(id, stageId) {
            const { data } = await api.put(`/opportunities/${id}/stage`, { stage_id: stageId });
            const index = this.opportunities.findIndex(o => o.id === id);
            if (index !== -1) this.opportunities[index] = data.data;
            return data.data;
        },

        async markWon(id) {
            const { data } = await api.post(`/opportunities/${id}/won`);
            const index = this.opportunities.findIndex(o => o.id === id);
            if (index !== -1) this.opportunities[index] = data.data;
            return data.data;
        },

        async markLost(id, reason) {
            const { data } = await api.post(`/opportunities/${id}/lost`, { reason });
            const index = this.opportunities.findIndex(o => o.id === id);
            if (index !== -1) this.opportunities[index] = data.data;
            return data.data;
        },
    },
});
