<?php

namespace Tests\Feature;

use App\Models\Campaign;
use App\Models\Contact;
use App\Models\Segment;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CampaignApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_create_campaign(): void
    {
        $data = [
            'name' => 'Welcome Email',
            'type' => 'email',
            'subject' => 'Welcome to Our Service',
            'content' => '<p>Thank you for joining!</p>',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/campaigns', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Welcome Email')
            ->assertJsonPath('data.status', 'draft');
    }

    public function test_can_send_campaign(): void
    {
        Contact::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'is_subscribed' => true,
        ]);
        
        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'draft',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/campaigns/{$campaign->id}/send");

        $response->assertOk();
        $this->assertEquals('sending', $campaign->fresh()->status);
    }

    public function test_can_schedule_campaign(): void
    {
        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'draft',
        ]);

        $scheduledAt = now()->addDays(7)->toDateTimeString();

        $response = $this->actingAs($this->user)
            ->postJson("/api/campaigns/{$campaign->id}/schedule", [
                'scheduled_at' => $scheduledAt,
            ]);

        $response->assertOk();
        $this->assertEquals('scheduled', $campaign->fresh()->status);
    }

    public function test_can_duplicate_campaign(): void
    {
        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'name' => 'Original Campaign',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/campaigns/{$campaign->id}/duplicate");

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Original Campaign (Copy)')
            ->assertJsonPath('data.status', 'draft');
    }

    public function test_can_get_campaign_stats(): void
    {
        $campaign = Campaign::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/campaigns/{$campaign->id}/stats");

        $response->assertOk()
            ->assertJsonStructure([
                'data' => ['total', 'delivered', 'opened', 'clicked', 'bounced'],
            ]);
    }
}
