<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ContactTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_contacts(): void
    {
        Contact::factory()->count(5)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/contacts');

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_contact(): void
    {
        $data = [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john@example.com',
            'phone' => '+97312345678',
            'company_name' => 'Acme Inc',
        ];

        $response = $this->actingAs($this->user)
            ->postJson('/api/contacts', $data);

        $response->assertCreated()
            ->assertJsonPath('data.first_name', 'John')
            ->assertJsonPath('data.email', 'john@example.com');

        $this->assertDatabaseHas('contacts', [
            'tenant_id' => $this->tenant->id,
            'email' => 'john@example.com',
        ]);
    }

    public function test_can_view_contact(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/contacts/{$contact->id}");

        $response->assertOk()
            ->assertJsonPath('data.id', $contact->id);
    }

    public function test_can_update_contact(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/contacts/{$contact->id}", [
                'first_name' => 'Updated Name',
            ]);

        $response->assertOk()
            ->assertJsonPath('data.first_name', 'Updated Name');
    }

    public function test_can_delete_contact(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->deleteJson("/api/contacts/{$contact->id}");

        $response->assertOk();
        $this->assertSoftDeleted('contacts', ['id' => $contact->id]);
    }

    public function test_cannot_access_other_tenant_contacts(): void
    {
        $otherTenant = Tenant::factory()->create();
        $contact = Contact::factory()->create(['tenant_id' => $otherTenant->id]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/contacts/{$contact->id}");

        $response->assertForbidden();
    }
}
