<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Invoice;
use App\Models\Opportunity;
use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Task;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DashboardApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_get_dashboard_stats(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Invoice::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'status' => 'sent',
            'total' => 1000.000,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/dashboard/stats');

        $response->assertOk()
            ->assertJsonStructure([
                'data' => [
                    'revenue',
                    'outstanding_invoices',
                    'contacts_count',
                    'tasks_due_today',
                ],
            ]);
    }

    public function test_can_get_recent_activities(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/dashboard/activities');

        $response->assertOk()
            ->assertJsonStructure(['data']);
    }

    public function test_can_get_revenue_chart_data(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/dashboard/revenue-chart');

        $response->assertOk()
            ->assertJsonStructure([
                'data' => [
                    'labels',
                    'revenue',
                    'expenses',
                ],
            ]);
    }

    public function test_can_get_pipeline_overview(): void
    {
        $pipeline = Pipeline::factory()->create(['tenant_id' => $this->tenant->id]);
        $stage = PipelineStage::factory()->create(['pipeline_id' => $pipeline->id]);
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Opportunity::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'contact_id' => $contact->id,
            'status' => 'open',
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/dashboard/pipeline');

        $response->assertOk()
            ->assertJsonStructure([
                'data' => [
                    'total_value',
                    'weighted_value',
                    'opportunities_count',
                ],
            ]);
    }
}
