<?php

namespace Tests\Feature;

use App\Models\Employee;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EmployeeTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_employees(): void
    {
        Employee::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => User::factory()->create(['tenant_id' => $this->tenant->id])->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/employees');

        $response->assertOk()->assertJsonCount(3, 'data');
    }

    public function test_can_create_employee(): void
    {
        $employeeUser = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $data = [
            'user_id' => $employeeUser->id,
            'hire_date' => now()->toDateString(),
            'job_title' => 'Software Developer',
            'employment_type' => 'full_time',
            'nationality' => 'BH',
            'basic_salary' => 1500.000,
            'housing_allowance' => 300.000,
        ];

        $response = $this->actingAs($this->user)->postJson('/api/employees', $data);

        $response->assertCreated()
            ->assertJsonPath('data.job_title', 'Software Developer')
            ->assertJsonPath('data.status', 'active');
    }

    public function test_can_update_employee_salary(): void
    {
        $employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => User::factory()->create(['tenant_id' => $this->tenant->id])->id,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/employees/{$employee->id}/salary", [
                'effective_date' => now()->toDateString(),
                'basic_salary' => 2000.000,
                'housing_allowance' => 400.000,
                'reason' => 'Annual increment',
            ]);

        $response->assertCreated();
    }

    public function test_can_terminate_employee(): void
    {
        $employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => User::factory()->create(['tenant_id' => $this->tenant->id])->id,
            'status' => 'active',
        ]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/employees/{$employee->id}", [
                'status' => 'terminated',
                'termination_date' => now()->toDateString(),
                'termination_reason' => 'Resignation',
            ]);

        $response->assertOk();
        $this->assertEquals('terminated', $employee->fresh()->status);
    }
}
