<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Invoice;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class InvoiceTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Contact $contact;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_invoices(): void
    {
        Invoice::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/invoices');

        $response->assertOk()
            ->assertJsonCount(3, 'data');
    }

    public function test_can_create_invoice(): void
    {
        $data = [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [
                [
                    'name' => 'Service A',
                    'quantity' => 1,
                    'unit_price' => 100.000,
                ],
                [
                    'name' => 'Service B',
                    'quantity' => 2,
                    'unit_price' => 50.000,
                ],
            ],
        ];

        $response = $this->actingAs($this->user)
            ->postJson('/api/invoices', $data);

        $response->assertCreated()
            ->assertJsonPath('data.contact_id', $this->contact->id)
            ->assertJsonPath('data.status', 'draft');

        $this->assertDatabaseHas('invoices', [
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
        ]);
    }

    public function test_invoice_calculates_totals_correctly(): void
    {
        $data = [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [
                ['name' => 'Item 1', 'quantity' => 2, 'unit_price' => 100.000],
                ['name' => 'Item 2', 'quantity' => 1, 'unit_price' => 50.000],
            ],
        ];

        $response = $this->actingAs($this->user)
            ->postJson('/api/invoices', $data);

        $response->assertCreated();
        
        $invoice = Invoice::find($response->json('data.id'));
        $this->assertEquals(250.000, $invoice->subtotal);
    }

    public function test_can_send_invoice(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'status' => 'draft',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/invoices/{$invoice->id}/send");

        $response->assertOk();
        
        $invoice->refresh();
        $this->assertEquals('sent', $invoice->status);
        $this->assertNotNull($invoice->sent_at);
    }

    public function test_can_record_payment(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'status' => 'sent',
            'total' => 500.000,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/invoices/{$invoice->id}/payment", [
                'payment_date' => now()->toDateString(),
                'amount' => 500.000,
                'payment_method' => 'bank_transfer',
            ]);

        $response->assertCreated();

        $invoice->refresh();
        $this->assertEquals(500.000, $invoice->amount_paid);
        $this->assertEquals('paid', $invoice->status);
    }

    public function test_cannot_update_paid_invoice(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'status' => 'paid',
        ]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/invoices/{$invoice->id}", [
                'notes' => 'Updated notes',
            ]);

        $response->assertStatus(422);
    }
}
