<?php

namespace Tests\Feature;

use App\Models\Employee;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class LeaveApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Employee $employee;
    protected LeaveType $leaveType;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->employee = Employee::factory()->create(['tenant_id' => $this->tenant->id, 'user_id' => $this->user->id]);
        $this->leaveType = LeaveType::factory()->create(['tenant_id' => $this->tenant->id, 'default_days' => 30]);
        LeaveBalance::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'entitled' => 30,
            'used' => 0,
        ]);
    }

    public function test_can_submit_leave_request(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/leave-requests', [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addDays(7)->toDateString(),
            'end_date' => now()->addDays(10)->toDateString(),
            'reason' => 'Family vacation',
        ]);

        $response->assertCreated()->assertJsonPath('data.status', 'pending');
    }

    public function test_can_approve_leave_request(): void
    {
        $leaveRequest = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'status' => 'pending',
            'days' => 3,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/leave-requests/{$leaveRequest->id}/approve");

        $response->assertOk();
        $this->assertEquals('approved', $leaveRequest->fresh()->status);
    }

    public function test_can_reject_leave_request(): void
    {
        $leaveRequest = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'status' => 'pending',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/leave-requests/{$leaveRequest->id}/reject", [
                'reason' => 'Team is understaffed',
            ]);

        $response->assertOk();
        $this->assertEquals('rejected', $leaveRequest->fresh()->status);
    }

    public function test_cannot_exceed_leave_balance(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/leave-requests', [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addDays(1)->toDateString(),
            'end_date' => now()->addDays(50)->toDateString(),
            'reason' => 'Long vacation',
        ]);

        $response->assertStatus(422);
    }

    public function test_can_get_leave_balance(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/leave-balances');

        $response->assertOk()
            ->assertJsonStructure(['data' => [['leave_type_id', 'entitled', 'used', 'remaining']]]);
    }
}
