<?php

namespace Tests\Feature;

use App\Models\Employee;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class LeaveTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Employee $employee;
    protected LeaveType $leaveType;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => $this->user->id,
        ]);
        $this->leaveType = LeaveType::factory()->create([
            'tenant_id' => $this->tenant->id,
            'days_per_year' => 30,
        ]);

        LeaveBalance::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'entitled' => 30,
            'used' => 0,
        ]);
    }

    public function test_can_submit_leave_request(): void
    {
        $data = [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addWeek()->toDateString(),
            'end_date' => now()->addWeek()->addDays(4)->toDateString(),
            'reason' => 'Annual vacation',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/leave/requests', $data);

        $response->assertCreated()
            ->assertJsonPath('data.status', 'pending');
    }

    public function test_cannot_request_more_than_balance(): void
    {
        LeaveBalance::where('employee_id', $this->employee->id)
            ->update(['used' => 28]);

        $data = [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addWeek()->toDateString(),
            'end_date' => now()->addWeek()->addDays(4)->toDateString(),
        ];

        $response = $this->actingAs($this->user)->postJson('/api/leave/requests', $data);

        $response->assertStatus(422);
    }

    public function test_can_approve_leave_request(): void
    {
        $request = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'days' => 3,
            'status' => 'pending',
        ]);

        $manager = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($manager)
            ->postJson("/api/leave/requests/{$request->id}/approve");

        $response->assertOk();
        $this->assertEquals('approved', $request->fresh()->status);
    }

    public function test_approving_leave_deducts_balance(): void
    {
        $request = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'days' => 5,
            'status' => 'pending',
        ]);

        $manager = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $this->actingAs($manager)->postJson("/api/leave/requests/{$request->id}/approve");

        $balance = LeaveBalance::where('employee_id', $this->employee->id)->first();
        $this->assertEquals(5, $balance->used);
    }

    public function test_can_cancel_leave_request(): void
    {
        $request = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'status' => 'pending',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/leave/requests/{$request->id}/cancel");

        $response->assertOk();
        $this->assertEquals('cancelled', $request->fresh()->status);
    }
}
