<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Opportunity;
use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OpportunityTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Contact $contact;
    protected Pipeline $pipeline;
    protected PipelineStage $stage;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->pipeline = Pipeline::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->stage = PipelineStage::factory()->create([
            'tenant_id' => $this->tenant->id,
            'pipeline_id' => $this->pipeline->id,
        ]);
    }

    public function test_can_list_opportunities(): void
    {
        Opportunity::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/opportunities');

        $response->assertOk()->assertJsonCount(3, 'data');
    }

    public function test_can_create_opportunity(): void
    {
        $data = [
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'name' => 'Big Deal',
            'value' => 50000,
            'probability' => 75,
            'expected_close_date' => now()->addMonth()->toDateString(),
        ];

        $response = $this->actingAs($this->user)->postJson('/api/opportunities', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Big Deal')
            ->assertJsonPath('data.value', 50000);
    }

    public function test_can_update_stage(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
        ]);

        $newStage = PipelineStage::factory()->create([
            'tenant_id' => $this->tenant->id,
            'pipeline_id' => $this->pipeline->id,
        ]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/opportunities/{$opportunity->id}/stage", [
                'stage_id' => $newStage->id,
            ]);

        $response->assertOk();
        $this->assertEquals($newStage->id, $opportunity->fresh()->stage_id);
    }

    public function test_can_mark_as_won(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'status' => 'open',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/opportunities/{$opportunity->id}/won");

        $response->assertOk();
        $opportunity->refresh();
        $this->assertEquals('won', $opportunity->status);
        $this->assertNotNull($opportunity->closed_at);
    }

    public function test_can_mark_as_lost(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'status' => 'open',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/opportunities/{$opportunity->id}/lost", [
                'lost_reason' => 'Budget constraints',
            ]);

        $response->assertOk();
        $opportunity->refresh();
        $this->assertEquals('lost', $opportunity->status);
        $this->assertEquals('Budget constraints', $opportunity->lost_reason);
    }
}
