<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Task;
use App\Models\TaskStatus;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProjectApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_projects(): void
    {
        Project::factory()->count(3)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/projects');

        $response->assertOk()
            ->assertJsonCount(3, 'data');
    }

    public function test_can_create_project(): void
    {
        $data = [
            'name' => 'Website Redesign',
            'description' => 'Complete website overhaul',
            'start_date' => now()->toDateString(),
            'due_date' => now()->addMonths(2)->toDateString(),
        ];

        $response = $this->actingAs($this->user)->postJson('/api/projects', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Website Redesign')
            ->assertJsonPath('data.status', 'active');
    }

    public function test_project_creates_default_statuses(): void
    {
        $data = [
            'name' => 'New Project',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/projects', $data);
        
        $project = Project::find($response->json('data.id'));
        $this->assertGreaterThan(0, $project->statuses()->count());
    }

    public function test_can_add_project_member(): void
    {
        $project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        $member = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/projects/{$project->id}/members", [
                'user_id' => $member->id,
                'role' => 'member',
            ]);

        $response->assertOk();
        $this->assertTrue($project->members()->where('user_id', $member->id)->exists());
    }

    public function test_can_archive_project(): void
    {
        $project = Project::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'active',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/projects/{$project->id}/archive");

        $response->assertOk();
        $this->assertEquals('archived', $project->fresh()->status);
    }

    public function test_project_progress_calculation(): void
    {
        $project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        $status = TaskStatus::factory()->create(['project_id' => $project->id, 'is_completed' => true]);
        $openStatus = TaskStatus::factory()->create(['project_id' => $project->id, 'is_completed' => false]);
        
        Task::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $project->id,
            'status_id' => $status->id,
            'completed_at' => now(),
        ]);
        Task::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $project->id,
            'status_id' => $openStatus->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/projects/{$project->id}");

        $response->assertOk()
            ->assertJsonPath('data.progress', 60);
    }
}
