<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProjectTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_projects(): void
    {
        Project::factory()->count(3)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/projects');

        $response->assertOk()->assertJsonCount(3, 'data');
    }

    public function test_can_create_project(): void
    {
        $data = [
            'name' => 'New Project',
            'description' => 'Project description',
            'color' => '#3B82F6',
            'start_date' => now()->toDateString(),
            'end_date' => now()->addMonths(3)->toDateString(),
        ];

        $response = $this->actingAs($this->user)->postJson('/api/projects', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'New Project')
            ->assertJsonPath('data.status', 'active');
    }

    public function test_can_add_member(): void
    {
        $project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        $member = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/projects/{$project->id}/members", [
                'user_id' => $member->id,
                'role' => 'member',
            ]);

        $response->assertOk();
        $this->assertTrue($project->members()->where('user_id', $member->id)->exists());
    }

    public function test_can_archive_project(): void
    {
        $project = Project::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'active',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/projects/{$project->id}/archive");

        $response->assertOk();
        $this->assertEquals('archived', $project->fresh()->status);
    }

    public function test_can_duplicate_project(): void
    {
        $project = Project::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/projects/{$project->id}/duplicate");

        $response->assertCreated();
        $this->assertEquals(2, Project::where('tenant_id', $this->tenant->id)->count());
    }
}
