<?php

namespace Tests\Feature;

use App\Models\Bill;
use App\Models\Contact;
use App\Models\Expense;
use App\Models\Invoice;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Vendor;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ReportApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_get_balance_sheet(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/reports/balance-sheet');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['assets', 'liabilities', 'equity']]);
    }

    public function test_can_get_profit_and_loss(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        Invoice::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'status' => 'paid',
        ]);
        Expense::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'approved',
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/reports/profit-loss');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['revenue', 'expenses', 'net_income']]);
    }

    public function test_can_get_cash_flow(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/reports/cash-flow');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['operating', 'investing', 'financing', 'net_change']]);
    }

    public function test_can_get_aging_receivables(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        Invoice::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'status' => 'sent',
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/reports/aging-receivables');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['items', 'totals']]);
    }

    public function test_can_get_aging_payables(): void
    {
        $vendor = Vendor::factory()->create(['tenant_id' => $this->tenant->id]);
        Bill::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'vendor_id' => $vendor->id,
            'status' => 'received',
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/reports/aging-payables');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['items', 'totals']]);
    }

    public function test_can_get_sales_report(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        Invoice::factory()->count(10)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/reports/sales');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['stats', 'invoices']]);
    }

    public function test_can_get_tax_report(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/reports/tax');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['output_vat', 'input_vat', 'vat_payable']]);
    }
}
