<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Task;
use App\Models\TaskStatus;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TaskApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Project $project;
    protected TaskStatus $status;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->status = TaskStatus::factory()->create(['project_id' => $this->project->id]);
    }

    public function test_can_create_task(): void
    {
        $data = [
            'title' => 'Implement user authentication',
            'description' => 'Add login and registration',
            'status_id' => $this->status->id,
            'priority' => 'high',
        ];

        $response = $this->actingAs($this->user)
            ->postJson("/api/projects/{$this->project->id}/tasks", $data);

        $response->assertCreated()
            ->assertJsonPath('data.title', 'Implement user authentication');
    }

    public function test_can_update_task_status(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'status_id' => $this->status->id,
        ]);

        $newStatus = TaskStatus::factory()->create(['project_id' => $this->project->id]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/tasks/{$task->id}/status", [
                'status_id' => $newStatus->id,
            ]);

        $response->assertOk();
        $this->assertEquals($newStatus->id, $task->fresh()->status_id);
    }

    public function test_can_complete_task(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'status_id' => $this->status->id,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/tasks/{$task->id}/complete");

        $response->assertOk();
        $this->assertNotNull($task->fresh()->completed_at);
    }

    public function test_can_add_comment_to_task(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'status_id' => $this->status->id,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/tasks/{$task->id}/comments", [
                'content' => 'This is a test comment',
            ]);

        $response->assertCreated();
        $this->assertDatabaseHas('task_comments', [
            'task_id' => $task->id,
            'content' => 'This is a test comment',
        ]);
    }

    public function test_can_assign_task(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'status_id' => $this->status->id,
        ]);

        $assignee = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/tasks/{$task->id}/assignees", [
                'user_id' => $assignee->id,
            ]);

        $response->assertOk();
        $this->assertTrue($task->assignees()->where('user_id', $assignee->id)->exists());
    }
}
