<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Task;
use App\Models\TaskStatus;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TaskTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Project $project;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        
        TaskStatus::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'name' => 'To Do',
            'is_default' => true,
        ]);
    }

    public function test_can_list_tasks(): void
    {
        Task::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/tasks?project_id=' . $this->project->id);

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_task(): void
    {
        $data = [
            'project_id' => $this->project->id,
            'title' => 'New Task',
            'description' => 'Task description',
            'priority' => 'high',
            'due_date' => now()->addWeek()->toDateString(),
        ];

        $response = $this->actingAs($this->user)
            ->postJson('/api/tasks', $data);

        $response->assertCreated()
            ->assertJsonPath('data.title', 'New Task')
            ->assertJsonPath('data.priority', 'high');
    }

    public function test_can_complete_task(): void
    {
        $completedStatus = TaskStatus::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'name' => 'Done',
            'is_completed' => true,
        ]);

        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/tasks/{$task->id}/complete");

        $response->assertOk();

        $task->refresh();
        $this->assertNotNull($task->completed_at);
    }

    public function test_can_add_comment(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/tasks/{$task->id}/comments", [
                'content' => 'This is a comment',
            ]);

        $response->assertCreated()
            ->assertJsonPath('data.content', 'This is a comment');
    }
}
