<?php

namespace Tests\Feature;

use App\Models\Bill;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Vendor;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class VendorApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_vendors(): void
    {
        Vendor::factory()->count(5)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/vendors');

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_vendor(): void
    {
        $data = [
            'name' => 'Office Supplies Co.',
            'email' => 'contact@supplies.com',
            'phone' => '+973 1234 5678',
            'payment_terms' => 'net_30',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/vendors', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Office Supplies Co.');
    }

    public function test_can_get_vendor_with_outstanding_balance(): void
    {
        $vendor = Vendor::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Bill::factory()->create([
            'tenant_id' => $this->tenant->id,
            'vendor_id' => $vendor->id,
            'total' => 1000.000,
            'amount_paid' => 400.000,
            'status' => 'partial',
        ]);

        $response = $this->actingAs($this->user)->getJson("/api/vendors/{$vendor->id}");

        $response->assertOk()
            ->assertJsonPath('data.outstanding_balance', 600.000);
    }

    public function test_can_search_vendors(): void
    {
        Vendor::factory()->create(['tenant_id' => $this->tenant->id, 'name' => 'ABC Company']);
        Vendor::factory()->create(['tenant_id' => $this->tenant->id, 'name' => 'XYZ Corp']);

        $response = $this->actingAs($this->user)->getJson('/api/vendors?search=ABC');

        $response->assertOk()
            ->assertJsonCount(1, 'data');
    }
}
