<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Workflow;
use App\Models\WorkflowEnrollment;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class WorkflowApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_create_workflow(): void
    {
        $data = [
            'name' => 'Welcome Sequence',
            'trigger_type' => 'contact_created',
            'steps' => [
                ['type' => 'send_email', 'config' => ['template_id' => 1], 'delay_minutes' => 0],
                ['type' => 'wait', 'config' => [], 'delay_minutes' => 1440],
                ['type' => 'add_tag', 'config' => ['tag_id' => 1], 'delay_minutes' => 0],
            ],
        ];

        $response = $this->actingAs($this->user)->postJson('/api/workflows', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Welcome Sequence')
            ->assertJsonCount(3, 'data.steps');
    }

    public function test_can_activate_workflow(): void
    {
        $workflow = Workflow::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => false,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/workflows/{$workflow->id}/activate");

        $response->assertOk();
        $this->assertTrue($workflow->fresh()->is_active);
    }

    public function test_can_enroll_contact_manually(): void
    {
        $workflow = Workflow::factory()->create([
            'tenant_id' => $this->tenant->id,
            'trigger_type' => 'manual',
            'is_active' => true,
        ]);

        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/workflows/{$workflow->id}/enroll", [
                'contact_id' => $contact->id,
            ]);

        $response->assertOk();
        $this->assertDatabaseHas('workflow_enrollments', [
            'workflow_id' => $workflow->id,
            'contact_id' => $contact->id,
        ]);
    }

    public function test_can_get_workflow_stats(): void
    {
        $workflow = Workflow::factory()->create(['tenant_id' => $this->tenant->id]);
        
        WorkflowEnrollment::factory()->count(10)->create([
            'tenant_id' => $this->tenant->id,
            'workflow_id' => $workflow->id,
            'status' => 'completed',
        ]);
        WorkflowEnrollment::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'workflow_id' => $workflow->id,
            'status' => 'active',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/workflows/{$workflow->id}/stats");

        $response->assertOk()
            ->assertJsonPath('data.total_enrolled', 15)
            ->assertJsonPath('data.completed', 10)
            ->assertJsonPath('data.active', 5);
    }
}
