<?php

namespace Tests\Unit;

use App\Models\Campaign;
use App\Models\Contact;
use App\Models\Segment;
use App\Models\Tenant;
use App\Services\CampaignService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CampaignServiceTest extends TestCase
{
    use RefreshDatabase;

    protected CampaignService $service;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = app(CampaignService::class);
        $this->tenant = Tenant::factory()->create();
    }

    public function test_creates_campaign(): void
    {
        $campaign = $this->service->create($this->tenant->id, [
            'name' => 'Welcome Email',
            'type' => 'email',
            'subject' => 'Welcome!',
            'content' => '<p>Hello!</p>',
        ]);

        $this->assertEquals('Welcome Email', $campaign->name);
        $this->assertEquals('draft', $campaign->status);
    }

    public function test_gets_recipients_from_segment(): void
    {
        $segment = Segment::factory()->create(['tenant_id' => $this->tenant->id]);
        $contacts = Contact::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'is_subscribed' => true,
        ]);
        $segment->contacts()->attach($contacts->pluck('id'));

        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'draft',
        ]);
        $campaign->segments()->attach($segment);

        $this->service->send($campaign);

        $this->assertEquals(5, $campaign->recipients()->count());
    }

    public function test_duplicates_campaign(): void
    {
        $original = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'name' => 'Original',
            'status' => 'sent',
        ]);

        $copy = $this->service->duplicate($original);

        $this->assertEquals('Original (Copy)', $copy->name);
        $this->assertEquals('draft', $copy->status);
    }

    public function test_schedules_campaign(): void
    {
        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'draft',
        ]);

        $scheduledAt = now()->addDays(7)->toDateTimeString();
        $this->service->schedule($campaign, $scheduledAt);

        $this->assertEquals('scheduled', $campaign->fresh()->status);
        $this->assertEquals($scheduledAt, $campaign->fresh()->scheduled_at);
    }

    public function test_calculates_stats(): void
    {
        $campaign = Campaign::factory()->create(['tenant_id' => $this->tenant->id]);
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $campaign->recipients()->create([
            'contact_id' => $contact->id,
            'email' => $contact->email,
            'status' => 'delivered',
            'opened_at' => now(),
        ]);

        $stats = $this->service->getStats($campaign);

        $this->assertEquals(1, $stats['total']);
        $this->assertEquals(1, $stats['delivered']);
        $this->assertEquals(1, $stats['opened']);
    }
}
