<?php

namespace Tests\Unit;

use App\Models\Employee;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\Tenant;
use App\Models\User;
use App\Services\LeaveService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class LeaveServiceTest extends TestCase
{
    use RefreshDatabase;

    protected LeaveService $service;
    protected Tenant $tenant;
    protected User $user;
    protected Employee $employee;
    protected LeaveType $leaveType;
    protected LeaveBalance $balance;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(LeaveService::class);
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => $this->user->id,
        ]);
        
        $this->leaveType = LeaveType::factory()->create([
            'tenant_id' => $this->tenant->id,
            'name' => 'Annual Leave',
            'days_per_year' => 30,
        ]);
        
        $this->balance = LeaveBalance::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'year' => now()->year,
            'entitled' => 30,
            'used' => 5,
            'carried_forward' => 0,
        ]);
    }

    public function test_creates_leave_request(): void
    {
        $data = [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addWeek()->toDateString(),
            'end_date' => now()->addWeek()->addDays(4)->toDateString(),
            'reason' => 'Vacation',
        ];

        $request = $this->service->request($this->employee, $data);

        $this->assertInstanceOf(LeaveRequest::class, $request);
        $this->assertEquals('pending', $request->status);
        $this->assertEquals(5, $request->days);
    }

    public function test_throws_exception_for_insufficient_balance(): void
    {
        $this->balance->update(['used' => 28]);

        $data = [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addWeek()->toDateString(),
            'end_date' => now()->addWeek()->addDays(4)->toDateString(),
        ];

        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('Insufficient leave balance');

        $this->service->request($this->employee, $data);
    }

    public function test_approves_leave_and_deducts_balance(): void
    {
        $request = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'days' => 3,
            'status' => 'pending',
        ]);

        $initialUsed = $this->balance->used;

        $this->service->approve($request, $this->user->id);

        $request->refresh();
        $this->balance->refresh();

        $this->assertEquals('approved', $request->status);
        $this->assertNotNull($request->approved_at);
        $this->assertEquals($initialUsed + 3, $this->balance->used);
    }

    public function test_cancel_restores_balance_for_approved_leave(): void
    {
        $request = LeaveRequest::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'leave_type_id' => $this->leaveType->id,
            'days' => 3,
            'status' => 'approved',
        ]);

        $this->balance->update(['used' => 8]);

        $this->service->cancel($request);

        $request->refresh();
        $this->balance->refresh();

        $this->assertEquals('cancelled', $request->status);
        $this->assertEquals(5, $this->balance->used);
    }

    public function test_calculates_half_days_correctly(): void
    {
        $data = [
            'leave_type_id' => $this->leaveType->id,
            'start_date' => now()->addWeek()->startOfWeek()->toDateString(),
            'end_date' => now()->addWeek()->startOfWeek()->addDays(2)->toDateString(),
            'start_half_day' => true,
            'end_half_day' => true,
        ];

        $request = $this->service->request($this->employee, $data);

        $this->assertEquals(2, $request->days);
    }
}
