<?php

namespace Tests\Unit;

use App\Models\Employee;
use App\Models\EmployeeSalary;
use App\Models\PayrollRun;
use App\Models\Tenant;
use App\Models\User;
use App\Services\PayrollService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class PayrollServiceTest extends TestCase
{
    use RefreshDatabase;

    protected PayrollService $service;
    protected Tenant $tenant;
    protected User $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(PayrollService::class);
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_creates_payroll_run(): void
    {
        $payrollRun = $this->service->create($this->tenant->id, [
            'pay_period_start' => '2024-01-01',
            'pay_period_end' => '2024-01-31',
            'payment_date' => '2024-01-31',
        ], $this->user->id);

        $this->assertInstanceOf(PayrollRun::class, $payrollRun);
        $this->assertEquals('draft', $payrollRun->status);
    }

    public function test_calculates_gosi_for_bahraini(): void
    {
        $employee = $this->createEmployee('BH', 1000);
        
        $payrollRun = $this->service->create($this->tenant->id, [
            'pay_period_start' => '2024-01-01',
            'pay_period_end' => '2024-01-31',
            'payment_date' => '2024-01-31',
        ], $this->user->id);

        $this->service->process($payrollRun);

        $item = $payrollRun->items()->where('employee_id', $employee->id)->first();
        
        // Bahraini: 7% employee, 12% employer
        $this->assertEquals(70.000, $item->gosi_employee);
        $this->assertEquals(120.000, $item->gosi_employer);
    }

    public function test_calculates_gosi_for_expat(): void
    {
        $employee = $this->createEmployee('IN', 1000);
        
        $payrollRun = $this->service->create($this->tenant->id, [
            'pay_period_start' => '2024-01-01',
            'pay_period_end' => '2024-01-31',
            'payment_date' => '2024-01-31',
        ], $this->user->id);

        $this->service->process($payrollRun);

        $item = $payrollRun->items()->where('employee_id', $employee->id)->first();
        
        // Expat: 1% employee, 3% employer
        $this->assertEquals(10.000, $item->gosi_employee);
        $this->assertEquals(30.000, $item->gosi_employer);
    }

    public function test_calculates_net_salary(): void
    {
        $employee = $this->createEmployee('BH', 1000);
        
        $payrollRun = $this->service->create($this->tenant->id, [
            'pay_period_start' => '2024-01-01',
            'pay_period_end' => '2024-01-31',
            'payment_date' => '2024-01-31',
        ], $this->user->id);

        $this->service->process($payrollRun);

        $item = $payrollRun->items()->where('employee_id', $employee->id)->first();
        
        // Gross: 1000, GOSI Employee: 70, Net: 930
        $this->assertEquals(1000.000, $item->gross_salary);
        $this->assertEquals(930.000, $item->net_salary);
    }

    protected function createEmployee(string $nationality, float $basicSalary): Employee
    {
        $user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        
        $employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => $user->id,
            'nationality' => $nationality,
            'status' => 'active',
        ]);

        EmployeeSalary::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $employee->id,
            'basic_salary' => $basicSalary,
            'housing_allowance' => 0,
            'transport_allowance' => 0,
            'other_allowances' => 0,
        ]);

        return $employee;
    }
}
